-- wtpanic_cooldown_report.lua
-- Sends a staff-wire message with GPS + location + facing direction when /wtpanic is on cooldown.

script_name('WTPanic Cooldown Reporter')
script_author('ncta')
script_version('2.0')

require 'lib.moonloader'

-- Optional (helps with zone helpers on some setups)
pcall(require, 'lib.sampfuncs')

local hasSampev, sampev = pcall(require, 'lib.samp.events')
if not hasSampev then sampev = nil end

-- ============================================================================
-- Settings
-- ============================================================================
local PANIC_CMD = '/wtpanic'
local PRIMARY_WT_CMD = '/wtalert'
local FALLBACK_WT_CMD = '/wt'

local COOLDOWN_NEEDLE = 'You must wait before using panic again.'
local NO_PERMISSION_NEEDLE = 'You do not have the permission to moderate this channel.'

-- How long after YOU type /wtpanic do we consider the cooldown line related to your attempt.
local PANIC_WINDOW_SEC = 6.0

-- If the server denies /wtalert, we fall back to /wt within this time window.
local ALERT_FALLBACK_WINDOW_SEC = 4.0

-- 8-direction compass output.
local COMPASS_DIRS = {
    'North', 'North-East', 'East', 'South-East',
    'South', 'South-West', 'West', 'North-West'
}

-- Movement-based auto-fix:
-- If heading source flips its rotation (rare, but happens on some setups after vehicle transitions),
-- we re-learn whether we should mirror (360 - heading) by comparing heading to your movement direction.
local AUTO_FIX_SPEED_THRESHOLD = 0.60 -- only learn when clearly moving (units/sec)
local AUTO_FIX_MAX_ACCEPT_DIFF = 35.0  -- degrees: only accept learn when movement roughly matches facing
local AUTO_FIX_MIN_IMPROVEMENT = 12.0  -- degrees: require meaningful improvement before switching

-- ============================================================================
-- Internal state
-- ============================================================================
local lastPanic = { id = 1, at = 0.0, pending = false }
local lastAlert = { id = 1, at = 0.0, msg = nil, waiting = false }

-- Heading quirks:
-- Some builds report ped heading in the opposite rotational direction,
-- which looks like East/West swapping. The previous auto-calibration
-- could occasionally lock the wrong mode after getting in/out of a car.
--
-- Fix strategy:
-- 1) Keep heading conversion deterministic (no permanent auto-calibration).
-- 2) Use car heading while in a vehicle (usually the most reliable).
-- 3) On-foot, apply a stable correction that matches most setups.
--
-- If your E/W is still swapped, you can toggle the correction at runtime
-- with /wtface (it will print the current mode).
-- Current correction modes (can change automatically if auto-fix has strong evidence)
local headingMode = {
    footMirrored = true,  -- most common fix for on-foot on many builds
    carMirrored = false,  -- cars are usually already consistent
}

-- For auto-fix learning
local lastPos = { x = nil, y = nil, z = nil, at = 0.0 }

-- ============================================================================
-- Small helpers
-- ============================================================================
local function nowSec()
    return os.clock()
end

local function contains_ic(hay, needle)
    if not hay or not needle then return false end
    hay = tostring(hay):lower()
    needle = tostring(needle):lower()
    return hay:find(needle, 1, true) ~= nil
end

local function sanitizeText(s)
    if not s then return nil end
    s = tostring(s)
    s = s:gsub('{[%x]+}', '')
    s = s:gsub('\r', ' '):gsub('\n', ' ')
    s = s:gsub('%s+', ' '):gsub('^%s+', ''):gsub('%s+$', '')
    return (s ~= '' and s) or nil
end

local function roundInt(x)
    if x == nil then return nil end
    x = tonumber(x)
    if not x then return nil end
    -- Proper rounding for negatives
    if x >= 0 then
        return math.floor(x + 0.5)
    else
        return math.ceil(x - 0.5)
    end
end

local function clampDeg(d)
    d = tonumber(d) or 0.0
    d = d % 360.0
    if d < 0 then d = d + 360.0 end
    return d
end

-- (angleDiff used to be needed for auto-calibration; kept intentionally unused-safe)
local function angleDiff(a, b)
    a, b = clampDeg(a), clampDeg(b)
    local d = math.abs(a - b)
    if d > 180 then d = 360 - d end
    return d
end

-- ============================================================================
-- Location (Region + Zone)
-- ============================================================================
local function getZoneNameAny(x, y, z)
    local ok, zone = pcall(function()
        -- SAMPFUNCS / MoonLoader helpers differ by build, so try a few.
        if type(getNameOfZone) == 'function' then
            local keyOrName = getNameOfZone(x, y, z)
            local cleaned = sanitizeText(keyOrName)
            if cleaned and type(getGxtText) == 'function' then
                local txt = sanitizeText(getGxtText(cleaned))
                if txt then return txt end
            end
            if cleaned then return cleaned end
        end

        if type(getZoneName) == 'function' then
            return sanitizeText(getZoneName(x, y, z))
        end

        if type(sampGetZoneName) == 'function' then
            return sanitizeText(sampGetZoneName(x, y, z))
        end

        return nil
    end)

    if not ok then return nil end
    return sanitizeText(zone)
end

local function guessRegionName(x, y, zone)
    -- Prefer using zone text if it already contains the region.
    local zl = (sanitizeText(zone) or ''):lower()
    if zl:find('san fierro', 1, true) then return 'San Fierro' end
    if zl:find('los santos', 1, true) then return 'Los Santos' end
    if zl:find('las venturas', 1, true) then return 'Las Venturas' end
    if zl:find('red county', 1, true) then return 'Red County' end
    if zl:find('flint county', 1, true) then return 'Flint County' end
    if zl:find('bone county', 1, true) then return 'Bone County' end
    if zl:find('tierra robada', 1, true) then return 'Tierra Robada' end
    if zl:find('whetstone', 1, true) then return 'Whetstone' end

    x, y = tonumber(x), tonumber(y)
    if not x or not y then return 'San Andreas' end

    -- Rough coordinate heuristics (good enough for city-level label).
    -- San Fierro
    if (x < -1270) and (y > -1115) and (y < 1659) then
        return 'San Fierro'
    end
    -- Los Santos
    if (x > 44) and (y < -768) then
        return 'Los Santos'
    end
    -- Las Venturas
    if (x > 869) and (y > 596) then
        return 'Las Venturas'
    end

    -- Counties / regions
    if (y > 596) and (x <= 869) then
        return 'Red County'
    end
    if (x < -1270) and (y < -1115) then
        return 'Whetstone'
    end
    if (x < -1270) and (y >= 1659) then
        return 'Tierra Robada'
    end
    if (x > 869) and (y >= -768) and (y <= 596) then
        return 'Bone County'
    end
    if (x >= -1270) and (x <= 869) and (y >= -768) and (y <= 596) then
        return 'Flint County'
    end

    return 'San Andreas'
end

local function formatLocation(region, zone)
    region = sanitizeText(region)
    zone = sanitizeText(zone)

    if zone and region then
        -- Avoid duplicates if zone already includes region.
        local zl, rl = zone:lower(), region:lower()
        if zl:find(rl, 1, true) then
            return zone
        end
        return string.format('%s, %s', region, zone)
    end

    return zone or region or 'Unknown'
end

-- ============================================================================
-- Facing direction
-- ============================================================================
local function getPedHeadingDeg(ped)
    if type(getCharHeading) == 'function' then
        local ok, h = pcall(getCharHeading, ped)
        if ok and h then return tonumber(h) end
    end
    if type(getHeading) == 'function' then
        local ok, h = pcall(getHeading, ped)
        if ok and h then return tonumber(h) end
    end
    return nil
end

local function getCarHeadingDeg(car)
    if type(getCarHeading) == 'function' then
        local ok, h = pcall(getCarHeading, car)
        if ok and h then return tonumber(h) end
    end
    return nil
end

local function normalizeHeading(rawDeg, mirrored)
    if rawDeg == nil then return nil end
    rawDeg = tonumber(rawDeg)
    if not rawDeg then return nil end
    if mirrored then
        return clampDeg(360.0 - rawDeg)
    end
    return clampDeg(rawDeg)
end

local function pickMirrorMode(rawDeg, moveCompassDeg)
    -- Returns true if mirrored is better, false otherwise, plus best diff.
    local a = normalizeHeading(rawDeg, false)
    local b = normalizeHeading(rawDeg, true)
    if not a or not b then return nil, nil end
    local dA = angleDiff(a, moveCompassDeg)
    local dB = angleDiff(b, moveCompassDeg)
    if dB < dA then return true, dB, dA end
    return false, dA, dB
end

local function getFacingDirection(ped)
    local heading = nil

    if type(isCharInAnyCar) == 'function' and type(storeCarCharIsInNoSave) == 'function' and isCharInAnyCar(ped) then
        local car = storeCarCharIsInNoSave(ped)
        local carH = getCarHeadingDeg(car)
        heading = normalizeHeading(carH, headingMode.carMirrored)
    else
        local pedH = getPedHeadingDeg(ped)
        if not pedH then return 'Unknown' end
        heading = normalizeHeading(pedH, headingMode.footMirrored)
    end

    if not heading then
        return 'Unknown'
    end

    -- Convert degrees to 8-way direction, standard compass.
    local idx = math.floor((clampDeg(heading) + 22.5) / 45.0) + 1
    if idx > 8 then idx = 1 end
    return COMPASS_DIRS[idx] or 'Unknown'
end

-- Movement-based auto-fix (learn whether foot/car heading should be mirrored)
local function updateHeadingModesFromMovement()
    local ped = PLAYER_PED
    if not ped or ped == 0 then return end
    if type(getCharCoordinates) ~= 'function' then return end

    local x, y, z = getCharCoordinates(ped)
    if not x or not y then return end

    local t = nowSec()
    if lastPos.x ~= nil and lastPos.y ~= nil and lastPos.at ~= 0.0 then
        local dt = t - lastPos.at
        if dt > 0.05 and dt < 5.0 then
            local dx = x - lastPos.x
            local dy = y - lastPos.y
            local dist = math.sqrt(dx * dx + dy * dy)
            local speed = dist / dt

            if speed >= AUTO_FIX_SPEED_THRESHOLD then
                -- Movement angle: 0=East, 90=North (math coords)
                local moveAngle = math.deg(math.atan2(dy, dx))
                if moveAngle < 0 then moveAngle = moveAngle + 360.0 end
                -- Convert to compass-like degrees with 0=North
                local moveCompass = clampDeg(90.0 - moveAngle)

                local inCar = (type(isCharInAnyCar) == 'function' and type(storeCarCharIsInNoSave) == 'function' and isCharInAnyCar(ped))
                if inCar then
                    local car = storeCarCharIsInNoSave(ped)
                    local raw = (car and car ~= 0) and getCarHeadingDeg(car) or nil
                    if raw then
                        local bestMirrored, bestDiff, otherDiff = pickMirrorMode(raw, moveCompass)
                        if bestMirrored ~= nil and bestDiff ~= nil and otherDiff ~= nil then
                            if bestDiff <= AUTO_FIX_MAX_ACCEPT_DIFF and (otherDiff - bestDiff) >= AUTO_FIX_MIN_IMPROVEMENT then
                                headingMode.carMirrored = bestMirrored
                            end
                        end
                    end
                else
                    local raw = getPedHeadingDeg(ped)
                    if raw then
                        local bestMirrored, bestDiff, otherDiff = pickMirrorMode(raw, moveCompass)
                        if bestMirrored ~= nil and bestDiff ~= nil and otherDiff ~= nil then
                            if bestDiff <= AUTO_FIX_MAX_ACCEPT_DIFF and (otherDiff - bestDiff) >= AUTO_FIX_MIN_IMPROVEMENT then
                                headingMode.footMirrored = bestMirrored
                            end
                        end
                    end
                end
            end
        end
    end

    lastPos.x, lastPos.y, lastPos.z, lastPos.at = x, y, z, t
end

-- ============================================================================
-- Message building + sending
-- ============================================================================
local function buildReportLine(x, y, z)
    local zone = getZoneNameAny(x, y, z)
    local region = guessRegionName(x, y, zone)
    local location = formatLocation(region, zone)

    local ped = PLAYER_PED
    local facing = (ped and ped ~= 0) and getFacingDirection(ped) or 'Unknown'

    local gx = roundInt(x) or 0
    local gy = roundInt(y) or 0

    -- Required format:
    -- "Panic cooldown - San Fierro, City Hall - Facing North. [GPS 714 -541]"
    return string.format('Panic cooldown - %s - Facing %s. [GPS %d %d]', location, facing, gx, gy)
end

local function markPanicAttempt(id)
    lastPanic.id = tonumber(id) or 1
    lastPanic.at = nowSec()
    lastPanic.pending = true
end

local function isRecentPanic()
    return lastPanic.pending and ((nowSec() - lastPanic.at) <= PANIC_WINDOW_SEC)
end

local function sendWirePrimaryThenFallbackIfNeeded(id, report)
    -- Try /wtalert first
    lastAlert.id = id
    lastAlert.msg = report
    lastAlert.at = nowSec()
    lastAlert.waiting = true

    sampSendChat(string.format('%s %d %s', PRIMARY_WT_CMD, id, report))
end

local function sendWireFallback()
    if not lastAlert.waiting then return end
    if (nowSec() - lastAlert.at) > ALERT_FALLBACK_WINDOW_SEC then
        lastAlert.waiting = false
        return
    end

    sampSendChat(string.format('%s %d %s', FALLBACK_WT_CMD, lastAlert.id or 1, lastAlert.msg or 'Panic cooldown'))
    lastAlert.waiting = false
end

local function handleCooldownTriggered()
    local ped = PLAYER_PED
    if not ped or ped == 0 then return end
    if type(getCharCoordinates) ~= 'function' then return end

    local x, y, z = getCharCoordinates(ped)
    if not x or not y then return end

    local report = buildReportLine(x, y, z or 0.0)
    sendWirePrimaryThenFallbackIfNeeded(lastPanic.id or 1, report)

    lastPanic.pending = false
end

-- ============================================================================
-- Events
-- ============================================================================
if sampev then
    function sampev.onSendCommand(cmd)
        if not cmd then return end
        local raw = tostring(cmd)
        local normalized = raw
        if normalized:sub(1, 1) ~= '/' then normalized = '/' .. normalized end

        if contains_ic(normalized, PANIC_CMD) then
            local id = normalized:match('^%s*/wtpanic%s+(%d+)%s*$')
            markPanicAttempt(id)
        end
    end

    function sampev.onServerMessage(color, text)
        if not text then return end

        -- Cooldown from /wtpanic
        if isRecentPanic() and contains_ic(text, COOLDOWN_NEEDLE) then
            handleCooldownTriggered()
            return
        end

        -- Permission denied for /wtalert -> send /wt fallback
        if lastAlert.waiting and contains_ic(text, NO_PERMISSION_NEEDLE) then
            sendWireFallback()
            return
        end
    end
else
    -- Fallback if lib.samp.events is missing: use /wtpc [id]
    function main()
        repeat wait(0) until isSampAvailable()

        sampRegisterChatCommand('wtface', function()
            headingMode.footMirrored = not headingMode.footMirrored
            local state = headingMode.footMirrored and 'ON' or 'OFF'
            sampAddChatMessage(string.format('{00FF90}[WTPanic]{FFFFFF} Foot mirror: %s', state), -1)
        end)

        sampRegisterChatCommand('wtfacecar', function()
            headingMode.carMirrored = not headingMode.carMirrored
            local state = headingMode.carMirrored and 'ON' or 'OFF'
            sampAddChatMessage(string.format('{00FF90}[WTPanic]{FFFFFF} Car mirror: %s', state), -1)
        end)

        sampRegisterChatCommand('wtpc', function(params)
            local id = tonumber(params) or 1
            markPanicAttempt(id)
            sampSendChat(string.format('%s %d', PANIC_CMD, id))
        end)

        sampAddChatMessage('{00FF90}[WTPanic]{FFFFFF} lib.samp.events not found. Use /wtpc [id] instead of /wtpanic [id].', -1)

        while true do
            wait(150)
            updateHeadingModesFromMovement()
        end
    end
    return
end

function main()
    repeat wait(0) until isSampAvailable()

    sampRegisterChatCommand('wtface', function()
        headingMode.footMirrored = not headingMode.footMirrored
        local state = headingMode.footMirrored and 'ON' or 'OFF'
        sampAddChatMessage(string.format('{00FF90}[WTPanic]{FFFFFF} Foot mirror: %s', state), -1)
    end)

    sampRegisterChatCommand('wtfacecar', function()
        headingMode.carMirrored = not headingMode.carMirrored
        local state = headingMode.carMirrored and 'ON' or 'OFF'
        sampAddChatMessage(string.format('{00FF90}[WTPanic]{FFFFFF} Car mirror: %s', state), -1)
    end)

    while true do
        wait(150)
        updateHeadingModesFromMovement()
    end
end