-- SAMP Lua Vehicle Tow Script (robust + single busy warning + debounce + safe auto-skip)
-- Only functional changes:
-- 1) Input mapping: single-vehicle = /vtow [groupIndex] vehiclemodel [index] [fix]
--    ALL = /vtowall [groupIndex] [fix]
-- 2) Step 3 now supports paginated Group lists (navigates to make want.groupIndex selectable).

-- Usage:
-- /vtow [groupIndex] vehiclemodel [index] [fix]
-- /vtowall [groupIndex] [fix]

script_name = "VehicleTowHelper"
script_author = "ncta"
script_version = "1.16.2-manual-vtow-safe2"

local sampev = require "lib.samp.events"

-- FSM/state
local towActive = false
local step = 0 -- 1..6
local want = { query = nil, fix = false, index = 1, groupIndex = 1, isAll = false, singleRepair = true, repairAnswer = true, mode = "GROUP", pickFirst = false, pickSlot = false, personalSlot = 1 }
local towed_vehicle_info = { name = "", id = "", nickname = "" }

local flow = "TOW" -- "PERKCHECK" or "TOW"

-- Behavior toggles
local DEBUG = false
local AUTO_SKIP_BUSY = true
local AUTO_RETRY_MS = 0
local DIALOG_TIMEOUT_MS = 4500
local lastDialogTick = 0
local awaitingRepairDialog = false
local lastMatchCount = nil
local isAutoSkipRun = false
local lastBusyMs = 0

-- Captured from server text for /vtowall (used only for our own chat line)
local lastAllTowedCount = nil
local lastAllTowedTotal = nil

-- Menu/faction text config
local CFG = {
    step1_line = "Tow individual vehicle",
    step2_line = "Group",
    step2_personal_line = "Personal",
    step3_line = "Mangano Crime Family", -- not used by logic, kept for compatibility
    yes_button_index = 1,
    dump_preview_lines = 25,

    -- Pagination button captions for Group list dialog (adjust if server differs)
    page_next_text = "Next",
    page_prev_text = "Back"
}

-- Utils
local function now_ms()
    -- Use a real wall-clock-ish timer. os.clock() can behave like CPU time and may not advance
    -- consistently in GTA/SA:MP.
    if type(getGameTimer) == "function" then
        return getGameTimer()
    end

    if not _G.__vtow_tick then
        local ok, ffi = pcall(require, "ffi")
        if ok then
            ffi.cdef[[unsigned long __stdcall GetTickCount(void);]]
            local k32 = ffi.load("kernel32")
            _G.__vtow_tick = function()
                return tonumber(k32.GetTickCount())
            end
        else
            _G.__vtow_tick = function()
                return math.floor(os.clock() * 1000)
            end
        end
    end

    return _G.__vtow_tick()
end

-- Suppress tow/repair server lines only for tows triggered by THIS script.
-- Manual /v tow will always show normally.
local SUPPRESS_TOW_SERVER_MSG_MS = 5000
local MANUAL_TOW_OVERRIDE_MS = 12000
local suppress_server_msgs_until = 0
local manual_tow_override_until = 0
local sending_script_vtow = false

local function mark_script_vtow_use()
    suppress_server_msgs_until = now_ms() + SUPPRESS_TOW_SERVER_MSG_MS
end

local function open_tow_menu()
    mark_script_vtow_use()
    sending_script_vtow = true
    sampSendChat("/v tow")
    sending_script_vtow = false
end
local function debug_print(message) if DEBUG then sampAddChatMessage(string.format("{00FFAA}[%s]{FFFFFF} %s", script_name, message), -1) end end
local function say(message) sampAddChatMessage(message, -1) end

-- Tow price display (chat only; does not affect logic)
local TOW_PRICE = 200
local REPAIR_PRICE = 2000
local TOW_REPAIR_PRICE = TOW_PRICE + REPAIR_PRICE

-- Personal vehicle tow pricing (chat only; does not affect logic)
local PTOW_PRICE = 2000
local PTOW_REPAIR_PRICE = 12000

local function format_money(n)
    n = tonumber(n) or 0
    local s = tostring(math.floor(n + 0.5))
    local sign = ""
    if s:sub(1, 1) == "-" then
        sign = "-"
        s = s:sub(2)
    end
    local out = s
    while true do
        local k
        out, k = out:gsub("^(%d+)(%d%d%d)", "%1,%2")
        if k == 0 then break end
    end
    return sign .. out
end

local function get_per_vehicle_cost(isRepair)
    if want.mode == "PERSONAL" then
        return isRepair and PTOW_REPAIR_PRICE or PTOW_PRICE
    end
    return isRepair and TOW_REPAIR_PRICE or TOW_PRICE
end

-- ========== PERK-AWARE SINGLE /VTOW ==========
-- We copy the same approach used in your tlist.lua:
-- 1) read /turfs to know what MCF owns
-- 2) read /turfperks all to know the perks
-- 3) if MCF has "repair group vehicles on spawn", we do tow only (no repair dialog)

local OUR_GROUP = "MCF"
local TARGET_REPAIR_PERK = "repair group vehicles on spawn"

local PERK_CHECK_CACHE_MS = 0 -- disabled (always refresh)
local PERK_CHECK_TIMEOUT_MS = 6500

local perk = {
    active = false,
    waitingTurfs = false,
    waitingPerks = false,
    lastCheckMs = 0,
    hasRepairOnSpawn = nil, -- true/false when known

    ownerIndexByTurf = {},
    perksByTurf = {},
}

local function strip_colors(s) return (s or ""):gsub("{%x%x%x%x%x%x}", "") end
local function normApos(s) return (s or ""):gsub("‘","'"):gsub("’","'") end
local function normSpaces(s) return (s or ""):gsub("\r",""):gsub("\t"," "):gsub("%s+"," "):gsub("^%s+",""):gsub("%s+$","") end
local function normName(s)
    s = strip_colors(s or "")
    s = normApos(s)
    s = s:gsub("\t"," ")
    s = s:lower()
    s = normSpaces(s)
    return s
end
local function eqGroup(a, b) return (tostring(a or ""):upper() == tostring(b or ""):upper()) end

local function normalizePerkText(s)
    s = normApos((s or ""):lower())
    s = s:gsub("pay ?[‘’']?n[‘’']? ?spray", "pay 'n' spray")
    s = s:gsub("discount on fuel and pay ?'n'? ?spray", "discount on fuel and pay 'n' spray")
    s = s:gsub("20%%+%s*time reduction for revives", "20% time reduction for revives")
    s = s:gsub("periodic%a* points for ownership", "periodical points for ownership")
    s = normSpaces(s)
    return s
end

local function tokenLooksLikeStatus(tok)
    tok = tostring(tok or "")
    if tok == "" then return true end
    if tok:match("^%d") then return true end
    if tok:match("^%[") or tok:match("^%(") then return true end
    if tok:match("%d+[hms]") then return true end
    if tok:match("Under") or tok:match("Attack") then return true end
    if tok:lower():find("now", 1, true) then return true end
    return false
end

local function parseTurfsForPerkCheck(text)
    perk.ownerIndexByTurf = {}
    text = strip_colors(text or "")
    for line in text:gmatch("[^\r\n]+") do
        if not line:find("Turf%s+Type") then
            local parts = {}
            for part in line:gmatch("%S+") do parts[#parts+1] = part end

            local globalIndex
            for i, v in ipairs(parts) do
                if v == "Global" then globalIndex = i break end
            end

            if globalIndex then
                local turfName = table.concat(parts, " ", 1, globalIndex - 1)
                local ownerName = parts[globalIndex + 1] or ""

                -- Some outputs still include an "alliance" column.
                local statusStartIndex = globalIndex + 2
                local maybe = parts[globalIndex + 2]
                if maybe and maybe ~= "" and not tokenLooksLikeStatus(maybe) then
                    statusStartIndex = globalIndex + 3
                end

                local capturableStatus = table.concat(parts, " ", statusStartIndex)
                local ownerFromTail = capturableStatus:match("%-%s*([%w_]+)%s*%(")
                if ownerFromTail and ownerFromTail ~= "" then ownerName = ownerFromTail end

                perk.ownerIndexByTurf[normName(turfName)] = { raw = turfName, owner = ownerName }
            end
        end
    end
end

local function join_wrapped_lines(lines)
    local joined = {}
    local buffer = ""
    for i = 1, #lines do
        local line = lines[i]
        if type(line) == "string" and line:find("\t", 1, true) then
            if buffer ~= "" then table.insert(joined, buffer) end
            buffer = line
        else
            if type(line) == "string" then
                if buffer == "" then buffer = line else buffer = buffer .. " " .. line end
            end
        end
    end
    if buffer ~= "" then
        if buffer:find("\t", 1, true) then
            table.insert(joined, buffer)
        else
            local nameGuess, perkGuess = buffer:match("^(.-)%s%s+(.+)$")
            if nameGuess and perkGuess then
                table.insert(joined, nameGuess .. "\t" .. perkGuess)
            end
        end
    end
    return joined
end

local function addPerk(turf, perkText)
    if type(turf) ~= "string" or type(perkText) ~= "string" then return end
    turf = normSpaces(turf)
    perkText = normSpaces(perkText)
    if turf == "" or perkText == "" then return end
    local key = normName(turf)
    perk.perksByTurf[key] = perk.perksByTurf[key] or {}
    table.insert(perk.perksByTurf[key], perkText)
end

local function parsePerksForPerkCheck(text)
    perk.perksByTurf = {}
    text = strip_colors(text or "")

    local rawLines = {}
    for raw in text:gmatch("[^\r\n]+") do
        local line = normApos(raw)
        if type(line) == "string" then table.insert(rawLines, line) end
    end

    local lines = join_wrapped_lines(rawLines)
    for i = 1, #lines do
        local raw = lines[i]
        if type(raw) == "string" then
            local nTab, pTab = raw:match("^(.-)\t(.+)$")
            if nTab and pTab then
                addPerk(nTab, pTab)
            else
                local nSp, pSp = raw:match("^(.-)%s%s+(.+)$")
                if nSp and pSp then
                    addPerk(nSp, pSp)
                else
                    local name1, perks1 = raw:match("^(.-)%s*%-%s*(.+)$")
                    if name1 and perks1 then
                        for perkTxt in perks1:gmatch("([^,]+)") do addPerk(name1, perkTxt) end
                    else
                        local name2, perks2 = raw:match("^(.-)%s*:%s*(.+)$")
                        if name2 and perks2 then
                            for perkTxt in perks2:gmatch("([^,]+)") do addPerk(name2, perkTxt) end
                        end
                    end
                end
            end
        end
    end
end

local function computeHasRepairOnSpawn()
    for key, meta in pairs(perk.ownerIndexByTurf) do
        if eqGroup(meta.owner, OUR_GROUP) then
            local list = perk.perksByTurf[key]
            if list then
                for _, p in ipairs(list) do
                    if normalizePerkText(p):find(TARGET_REPAIR_PERK, 1, true) then
                        return true
                    end
                end
            end
        end
    end
    return false
end

local function startTowFlow()
    flow = "TOW"
    step = 1
    open_tow_menu()
    debug_print(("Step 0: Sent '/v tow' (single repair %s)"):format(want.singleRepair and "ON" or "OFF"))
    lastDialogTick = now_ms()
end

local function startPerkCheck()
    -- Simplified perk check: only opens /turfperks and searches for the perk text.
    perk.active = true
    perk.waitingTurfs = false
    perk.waitingPerks = true
    perk.ownerIndexByTurf = {}
    perk.perksByTurf = {}

    lastDialogTick = now_ms()
    sampSendChat("/turfperks")
end

local function finishPerkCheckAndStartTow(hasPerk, note)
    perk.active = false
    perk.waitingTurfs = false
    perk.waitingPerks = false

    perk.hasRepairOnSpawn = (hasPerk == true)
    perk.lastCheckMs = now_ms()

    -- If the user explicitly typed 'fix', always do tow + repair.
    -- Otherwise, if perk is active, tow only. If perk is not active, tow + repair.
    if want.fix == true then
        want.singleRepair = true
    else
        want.singleRepair = not perk.hasRepairOnSpawn
    end

    if note and note ~= "" then
        debug_print(("Perk check result: %s (note: %s)"):format(tostring(perk.hasRepairOnSpawn), note))
    else
        debug_print(("Perk check result: %s"):format(tostring(perk.hasRepairOnSpawn)))
    end

    startTowFlow()
end

local function perkCheckTimeoutFallback()
    -- If we cannot detect, keep old behavior for single vehicles: tow + repair.
    finishPerkCheckAndStartTow(false, "timeout fallback")
end


local function collect_lines(text)
    local t = {}
    for line in text:gmatch("[^\r\n]+") do
        if line:match("%S") then
            t[#t+1] = line
        end
    end
    return t
end

local function find_row(text, query)
    local lines = collect_lines(text)
    for i, l in ipairs(lines) do
        if l:lower():find(query:lower(), 1, true) then
            return i-1
        end
    end
    return nil
end

local function find_vehicle_idx(text, query, target_index)
    local search = query:lower()
    local lines = collect_lines(text)
    local found, matches = -1, 0
    for i, line in ipairs(lines) do
        local left = line:match("^([^%s]+)")
        local right = line:match("%s+(.+)$")
        if (left and left:lower() == search) or (right and right:lower() == search) then
            matches = matches + 1
            if matches == target_index then
                found = i-1
                towed_vehicle_info.name = left or ""
                towed_vehicle_info.nickname = right or "None"
                local idm = line:match("\t(%d+)\t") or line:match("\t(%d+)$")
                towed_vehicle_info.id = idm or "Unknown"
                break
            end
        end
    end
    return found, matches
end

local function dump_dialog_preview(text, msg)
    debug_print(msg)
    local n = 0
    for line in text:gmatch("[^\r\n]+") do
        debug_print(line)
        n=n+1
        if n>=CFG.dump_preview_lines then break end
    end
end

local function reset_flow(reason)
    if reason then debug_print("Reset flow: " .. reason) end
    towActive=false; step=0; awaitingRepairDialog=false; lastDialogTick=0; lastMatchCount=nil; isAutoSkipRun=false
    lastAllTowedCount=nil; lastAllTowedTotal=nil
    flow = "TOW"
    want.mode = "GROUP"
    want.pickFirst = false
    want.pickSlot = false
    want.personalSlot = 1
    perk.active=false; perk.waitingTurfs=false; perk.waitingPerks=false
end

local function warn_vehicle_in_use()
    if displayGameText then displayGameText("~r~VEHICLE IN USE~n~~w~Tow aborted",3000,4) return end
    if sampfuncs and sampfuncs.displayGameText then sampfuncs.displayGameText("~r~VEHICLE IN USE~n~~w~Tow aborted",3000,4) return end
    local labelName = towed_vehicle_info.name ~= "" and towed_vehicle_info.name or "Vehicle"
    local labelNick = towed_vehicle_info.nickname ~= "" and towed_vehicle_info.nickname or "Unknown"
    sampAddChatMessage(("{FF5555}Vehicle{FFFFFF} %s (%s) {FF5555}is in use and cannot be towed."):format(labelName,labelNick), -1)
end


-- Helpers for personal tow selection (slot-based)
local function set_vehicle_info_from_line(line)
    line = line or ""
    local left = line:match("^([^%s]+)") or ""
    local right = line:match("%s+(.+)$") or "None"
    local idm = line:match("\t(%d+)\t") or line:match("\t(%d+)$") or "Unknown"
    towed_vehicle_info.name = left
    towed_vehicle_info.nickname = (right ~= "" and right) or "None"
    towed_vehicle_info.id = idm
end

local function find_first_vehicle_row(text)
    local lines = collect_lines(text)
    for i, l in ipairs(lines) do
        -- Vehicle rows include an ID column (digits after a tab). Header rows typically do not.
        if l:match("\t%d") then
            return i-1, l
        end
    end
    return 0, lines[1] or ""
end

local function find_nth_vehicle_row(text, n)
    n = tonumber(n) or 1
    if n < 1 then n = 1 end
    local lines = collect_lines(text)

    local count = 0
    local hadIdColumn = false

    for i, l in ipairs(lines) do
        -- Prefer rows that include an ID column (digits after a tab).
        if l:match("\t%d") then
            hadIdColumn = true
            count = count + 1
            if count == n then
                return i-1, l
            end
        end
    end

    -- If the dialog does include IDs but we didn't reach n, it's out of range.
    if hadIdColumn then
        return -1, ""
    end

    -- Fallback: some servers show personal vehicles without an ID column.
    if n <= #lines then
        return n-1, lines[n] or ""
    end

    return -1, ""
end


-- Public commands
function cmd_vtow(params)
    if towActive and lastDialogTick > 0 and (now_ms() - lastDialogTick) > (DIALOG_TIMEOUT_MS + 200) then
        reset_flow("auto-reset stale")
    end
    if towActive then debug_print("Tow already in progress, please wait..."); return end

    local t = {}; for w in params:gmatch("%S+") do t[#t+1]=w end
    if #t < 1 then
        debug_print("Usage: /vtow [groupIndex] vehiclemodel [index] [fix] | /vtowall [groupIndex] [fix]")
        return
    end

    want.mode = "GROUP"; want.isAll=false; want.query=nil; want.index=1; want.fix=false; want.groupIndex=1; want.repairAnswer=true; isAutoSkipRun=false
    lastAllTowedCount = nil
    lastAllTowedTotal = nil

    local first = t[1]:lower()

    -- /vtow all [groupIndex] [fix] (deprecated; use /vtowall)
    if first == "all" then
        debug_print("Note: '/vtow all' is deprecated. Use /vtowall [groupIndex] [fix].")
        local rest = params:match("^%s*%S+%s*(.*)$") or ""
        cmd_vtowall(rest)
        return
    end

    -- Single vehicle: [groupIndex] vehiclemodel [index] [fix]
    local maybeGroup = tonumber(t[1])
    local idx = 1
    if maybeGroup then
        want.groupIndex = math.max(1, math.floor(maybeGroup))
        if #t < 2 then
            debug_print("Usage: /vtow [groupIndex] vehiclemodel [index] [fix]")
            return
        end
        want.query = t[2]:lower()
        idx = 3
    else
        want.groupIndex = 1
        want.query = t[1]:lower()
        idx = 2
    end

    for i=idx,#t do
        local a = t[i]:lower()
        if tonumber(a) then
            want.index = tonumber(a)
        elseif a == "fix" then
            -- Kept for backwards compatibility. Single /vtow uses perk-aware repair now.
            want.fix = true
        end
    end

    debug_print(("Tow '%s' idx %d in group #%d (perk-aware repair)"):format(
        want.query, want.index, want.groupIndex))

    -- Decide single /vtow repair based on current MCF turf perks:
    -- If MCF has "repair group vehicles on spawn" active, do tow only.
    -- Otherwise do tow + repair (old behavior).
    want.singleRepair = true
    towActive = true
    lastDialogTick = now_ms()

    -- Always refresh perk state (turf perks can change; cache caused wrong behavior)
    flow = "PERKCHECK"
    step = 0
    startPerkCheck()
    debug_print("Perk check started: /turfperks")
end

function cmd_vtowall(params)
    if towActive and lastDialogTick > 0 and (now_ms() - lastDialogTick) > (DIALOG_TIMEOUT_MS + 200) then
        reset_flow("auto-reset stale")
    end
    if towActive then
        debug_print("Tow already in progress, please wait...")
        return
    end

    local t = {}; for w in (params or ""):gmatch("%S+") do t[#t+1] = w end

    want.mode = "GROUP"
    want.isAll = true
    want.query = "all"
    want.index = 1
    want.fix = false
    want.groupIndex = 1
    want.repairAnswer = true
    isAutoSkipRun = false
    lastAllTowedCount = nil
    lastAllTowedTotal = nil

    local tmpGroup = nil
    for i = 1, #t do
        local a = t[i]:lower()
        if tonumber(a) then
            tmpGroup = tonumber(a)
        elseif a == "fix" then
            want.fix = true
        end
    end
    want.groupIndex = tmpGroup or 1

    debug_print(("Tow ALL in group #%d%s"):format(want.groupIndex, want.fix and " with repair" or ""))
    flow = "TOW"

    step = 1
    towActive = true
    open_tow_menu()
    debug_print("Step 0: Sent '/v tow'")
    lastDialogTick = now_ms()
end

-- Personal vehicle towing
-- /ptow vehiclemodel [index] [fix]
-- /pvtow vehiclemodel [index] [fix]
function cmd_ptow(params)
    if towActive and lastDialogTick > 0 and (now_ms() - lastDialogTick) > (DIALOG_TIMEOUT_MS + 200) then
        reset_flow("auto-reset stale")
    end
    if towActive then debug_print("Tow already in progress, please wait..."); return end

    local t = {}; for w in (params or ""):gmatch("%S+") do t[#t+1]=w end
    if #t < 1 then
        debug_print("Usage: /ptow <slot> [fix] | /ptow vehiclemodel [index] [fix] | /ptow all [fix] | /ptowall [fix]")
        return
    end

    want.mode = "PERSONAL"
    want.groupIndex = 1
    want.repairAnswer = false
    want.singleRepair = false
    want.pickFirst = false
    want.pickSlot = false
    want.personalSlot = 1
    isAutoSkipRun = false
    lastAllTowedCount = nil
    lastAllTowedTotal = nil

    local first = t[1]:lower()
    -- /ptow all (or /ptow al) => tow ALL personal vehicles
    if first == "all" or first == "al" then
        want.isAll = true
        want.query = "all"
        want.index = 1
        want.fix = false
        for i = 2, #t do
            local a = t[i]:lower()
            if a == "fix" then want.fix = true end
        end
        debug_print(("Personal tow ALL%s"):format(want.fix and " with repair" or ""))

    -- /ptow <slot> [fix] => personal slot number (1 = first, 2 = second, etc.)
    elseif tonumber(first) then
        want.isAll = false
        want.pickFirst = false
        want.pickSlot = true
        want.personalSlot = math.max(1, math.floor(tonumber(first)))
        want.query = ""
        want.index = want.personalSlot
        want.fix = false
        for i = 2, #t do
            local a = t[i]:lower()
            if a == "fix" then want.fix = true end
        end
        debug_print(("Personal tow SLOT %d%s"):format(want.personalSlot, want.fix and " with repair" or ""))

    -- /ptow model [index] [fix] (same matching style as group tow)
    else
        want.isAll = false
        want.query = first
        want.index = 1
        want.fix = false
        for i = 2, #t do
            local a = t[i]:lower()
            if tonumber(a) then
                want.index = tonumber(a)
            elseif a == "fix" then
                want.fix = true
            end
        end
        debug_print(("Personal tow '%s' idx %d%s"):format(want.query, want.index, want.fix and " with repair" or ""))
    end

    flow = "TOW"
    step = 1
    towActive = true
    open_tow_menu()
    debug_print("Step 0: Sent '/v tow' (personal)")
    lastDialogTick = now_ms()
end

-- Personal tow ALL aliases
function cmd_ptowall(params)
    local p = params or ""
    if p ~= "" then p = " " .. p end
    cmd_ptow("all" .. p)
end

function cmd_vtowdebug()
    DEBUG = not DEBUG
    sampAddChatMessage(string.format("{00FFAA}[%s]{FFFFFF} Debug %s", script_name, DEBUG and "enabled" or "disabled"), -1)
end

-- Helper: detect pagination buttons in dialog text block
local function has_button_label(text, label)
    if not label or label == "" then return false end
    return text:lower():find(label:lower(), 1, true) ~= nil
end


-- Detect manual /v tow so we never suppress its server messages.
function sampev.onSendCommand(cmd)
    local c = (cmd or ""):lower()
    -- Depending on MoonLoader/SAMP events, cmd may include a leading "/" or not.
    -- Treat any manual "/v tow" as an override so we never suppress its server messages.
    if c:match("^%s*/?v%s+tow") then
        if not sending_script_vtow then
            manual_tow_override_until = now_ms() + MANUAL_TOW_OVERRIDE_MS
            -- Also clear any pending suppression window from a previous scripted tow,
            -- so manual tows right after a /vtow are never affected.
            suppress_server_msgs_until = 0
        end
    end
end
-- Event hooks
function sampev.onShowDialog(id, style, title, btn1, btn2, text)
    if not towActive then return end
    lastDialogTick = now_ms()

    -- PERK CHECK (single /vtow only)
    if flow == "PERKCHECK" then
        local titleClean = strip_colors(title or "")
        local tl = titleClean:lower()

        -- We only care about the /turfperks dialog. When it appears, close it and scan for the perk.
        if perk.waitingPerks and tl:find("turf", 1, true) and (tl:find("perk", 1, true) or tl:find("perks", 1, true)) then
            perk.waitingPerks = false
            local closeBtn = 1
            if btn2 and btn2:lower():find("close", 1, true) then closeBtn = 0 end
            sampSendDialogResponse(id, closeBtn, 0, "")

            local hay = normalizePerkText(strip_colors(text or ""))
            local hasPerk = (hay:find(TARGET_REPAIR_PERK, 1, true) ~= nil)

            finishPerkCheckAndStartTow(hasPerk, "turfperks scan")
            return false
        end

        -- Not our dialog, do not consume it.
        return
    end

    -- STEP 1
    if step == 1 then
        local option = want.isAll and "Tow all vehicles" or CFG.step1_line
        local row = find_row(text, option) or (want.isAll and 1 or 0)
        debug_print(("Step1: pick \"%s\" -> %d"):format(option, row))
        sampSendDialogResponse(id, 1, row, "")
        step = 2
        return false
    end

    -- STEP 2
    if step == 2 then
        local pickText = (want.mode == "PERSONAL") and CFG.step2_personal_line or CFG.step2_line
        local row = find_row(text, pickText)

        -- If we can't find it, fall back to first option for PERSONAL and second option for GROUP.
        if row == nil then
            if want.mode == "PERSONAL" then
                row = 0
            else
                row = 1
            end
        end

        debug_print(("Step2: pick \"%s\" -> %d"):format(pickText, row))
        sampSendDialogResponse(id, 1, row, "")

        -- PERSONAL goes straight to vehicle list (no group list step)
        step = (want.mode == "PERSONAL") and 4 or 3
        return false
    end

    -- STEP 3: choose numeric group (with pagination support)
    if step == 3 then
        local lines = collect_lines(text)

        local totalLines = #lines
        local pageHasNext = has_button_label(text, CFG.page_next_text)
        local pageHasPrev = has_button_label(text, CFG.page_prev_text)

        local navReserve = 0
        if pageHasNext or pageHasPrev then
            navReserve = (pageHasNext and pageHasPrev) and 2 or 1
        end

        local visibleDataCount = math.max(0, totalLines - navReserve)

        local targetIndex = want.groupIndex
        if targetIndex <= visibleDataCount and targetIndex > 0 then
            local targetRow = targetIndex - 1
            debug_print(("Step3: pick group #%d (row %d) on current page"):format(want.groupIndex, targetRow))
            sampSendDialogResponse(id, 1, targetRow, "")
            step = 4
            return false
        end

        if pageHasNext then
            debug_print(("Step3: paginate Next to reach group #%d (visible %d)"):format(want.groupIndex, visibleDataCount))
            local nextRow = totalLines - 1
            sampSendDialogResponse(id, 1, nextRow, "")
            return false
        end

        dump_dialog_preview(text, ("Step3: group index %d not reachable; no more pages"):format(want.groupIndex or -1))
        say(("{FF5555}Failed: group index %d not available."):format(want.groupIndex or -1))
        reset_flow("group index unreachable")
        return false
    end

    -- STEP 4
    if step == 4 then
        if want.isAll then
            debug_print("Step4: select anything (index 0) for ALL")
            sampSendDialogResponse(id, 1, 0, "")
            step = 5
            return false
        end


        -- /ptow <slot>: pick personal vehicle slot (counts only actual vehicle rows, so headers do not break selection)
        if want.mode == "PERSONAL" and want.pickSlot then
            local slot = want.personalSlot or want.index or 1
            local rowPick, linePick = find_nth_vehicle_row(text, slot)
            if rowPick == -1 then
                dump_dialog_preview(text, ("Step4: personal slot %d not available"):format(slot))
                say(("{FF5555}Personal vehicle slot %d not available."):format(slot))
                reset_flow("personal slot not found")
                return false
            end
            debug_print(("Step4: pick personal slot %d -> %d"):format(slot, rowPick))
            set_vehicle_info_from_line(linePick)
            sampSendDialogResponse(id, 1, rowPick, "")
            step = 5
            return false
        end

        -- Legacy: pick the first personal vehicle from the list (kept for compatibility)
        if want.mode == "PERSONAL" and want.pickFirst then
            local firstRow, firstLine = find_first_vehicle_row(text)
            debug_print(("Step4: pick first personal vehicle -> %d"):format(firstRow))
            set_vehicle_info_from_line(firstLine)
            sampSendDialogResponse(id, 1, firstRow, "")
            step = 5
            return false
        end
        local row, match_count = find_vehicle_idx(text, want.query, want.index)
        lastMatchCount = match_count
        if row == -1 then
            if isAutoSkipRun then reset_flow("auto-skip overflow") return false end
            if match_count > 0 then
                say(("{FF5555}Found %d matches for '%s', but index %d doesn't exist. Use 1-%d."):format(
                    match_count, want.query, want.index, match_count))
            else
                say(("{FF5555}Vehicle or nickname '%s' not found in the list."):format(want.query))
            end
            dump_dialog_preview(text, ("Step4: vehicle/nickname \"%s\" (index %d) not found"):format(want.query, want.index))
            reset_flow("vehicle row not found")
            return false
        end

        debug_print(("Step4: select '%s' (idx %d) -> menu row %d"):format(want.query, want.index, row))
        sampSendDialogResponse(id, 1, row, "")
        step = 5
        return false
    end

    -- STEP 5
    if step == 5 then
        debug_print("Step5: confirm tow -> Yes")
        sampSendDialogResponse(id, CFG.yes_button_index, -1, "")
        if want.isAll then
            -- /vtowall always shows repair confirm on this server.
            -- If user passed 'fix', press Yes, otherwise press No.
            want.repairAnswer = (want.fix == true)
        elseif want.mode == "PERSONAL" then
            -- Personal towing is independent of turf perks; 'fix' decides repair.
            want.repairAnswer = (want.fix == true)
        else
            -- Single /vtow: press Yes only if singleRepair is enabled (perk not active).
            want.repairAnswer = (want.singleRepair == true)
        end
        awaitingRepairDialog = true
        step = 6
        lastDialogTick = now_ms()
        return false
    end

    -- STEP 6
    if step == 6 then
        local btn1ok = btn1 and btn1:lower():find("yes")
        local btn2ok = btn2 and btn2:lower():find("no")
        if not (btn1ok and btn2ok) then
            debug_print("Unexpected dialog at Step6; aborting")
            reset_flow("unexpected dialog at repair")
            return false
        end

        local answerYes = (want.repairAnswer == true)
        debug_print(("Step6: repair confirm -> %s"):format(answerYes and "Yes" or "No"))
        -- button 1 = left button (Yes), button 0 = right button (No)
        sampSendDialogResponse(id, answerYes and CFG.yes_button_index or 0, -1, "")

        local perVehicleCost = get_per_vehicle_cost(answerYes)

        if want.isAll then
            local count = tonumber(lastAllTowedCount)
            local total = tonumber(lastAllTowedTotal)
            local totalCost = count and (count * perVehicleCost) or nil
            local base = (count and total) and ("{00FF00}" .. tostring(count) .. "/" .. tostring(total) .. " vehicles")
                or (count and ("{00FF00}" .. tostring(count) .. " vehicles") or "{00FF00}All vehicles")

            if answerYes then
                if totalCost then
                    say(("%s {00FF00}have been towed and repaired for {FFFFFF}$%s{00FF00} ({FFFFFF}$%s{00FF00} each).")
                        :format(base, format_money(totalCost), format_money(perVehicleCost)))
                else
                    say(("%s {00FF00}have been towed and repaired ({FFFFFF}$%s{00FF00} each).")
                        :format(base, format_money(perVehicleCost)))
                end
            else
                if totalCost then
                    say(("%s {00FF00}have been towed for {FFFFFF}$%s{00FF00} ({FFFFFF}$%s{00FF00} each).")
                        :format(base, format_money(totalCost), format_money(perVehicleCost)))
                else
                    say(("%s {00FF00}have been towed ({FFFFFF}$%s{00FF00} each).")
                        :format(base, format_money(perVehicleCost)))
                end
            end
        else
            if answerYes then
                say(("{00FF00}Vehicle {FFFFFF}%s (%s) {00FF00}has been towed and repaired for {FFFFFF}$%s{00FF00}."):format(
                    towed_vehicle_info.name, towed_vehicle_info.nickname, format_money(perVehicleCost)))
            else
                say(("{00FF00}Vehicle {FFFFFF}%s (%s) {00FF00}has been towed for {FFFFFF}$%s{00FF00}."):format(
                    towed_vehicle_info.name, towed_vehicle_info.nickname, format_money(perVehicleCost)))
            end
        end

        reset_flow("repair dialog handled")
        return false
    end

    dump_dialog_preview(text, "Unexpected dialog during tow")
    return false
end

-- Server messages
function sampev.onServerMessage(color, msg)
        -- Extra safety: suppress specific tow/repair server lines even if our internal tow flow has already reset.
    -- (Some servers send these a bit later than our own custom message.)
    msg = msg:gsub("{%x%x%x%x%x%x}", "") -- strip SA:MP color tags if present
    local lower = msg:lower()
local inScriptContext = towActive or (now_ms() < suppress_server_msgs_until)
local manualOverride = (now_ms() < manual_tow_override_until)

if inScriptContext and not manualOverride then
    -- Single tow+repair line (we print our own combined $ message)
    if lower:find("selected vehicle is successfully repaired for", 1, true) then
        return false
    end

    -- Group tow summary line (we print our own totals)
    do
        local a, b = msg:match("^(%d+)%s+out%s+of%s+(%d+)%s+selected vehicles have been towed")
        if a and b then
            lastAllTowedCount = tonumber(a)
            lastAllTowedTotal = tonumber(b)
            return false
        end
    end

    -- Group repair summary line (some servers send this separately)
    do
        local a, b = msg:match("^(%d+)%s+out%s+of%s+(%d+)%s+selected vehicles have been repaired")
        if a and b then
            lastAllTowedCount = tonumber(a)
            lastAllTowedTotal = tonumber(b)
            return false
        end
    end

    -- Late single-tow line (some servers send this a bit after dialogs finish)
    if not towActive and lower:find("selected vehicle is successfully towed", 1, true) then
        return false
    end
end

-- From here on, only react to messages while our tow flow is active
    if not towActive then return end
    if flow ~= "TOW" then return end
if lower:find("unable to tow this vehicle as it's currently being used", 1, true) then
        local tnow = now_ms(); if (tnow - lastBusyMs) < 300 then return false end; lastBusyMs = tnow
        debug_print("Tow failed: vehicle busy")
        local prev_query, prev_index, prev_fix, prev_group = want.query, want.index, want.fix, want.groupIndex
        local maxCount = lastMatchCount
        reset_flow("vehicle busy")
        warn_vehicle_in_use()

        if not want.isAll then
            if AUTO_SKIP_BUSY and maxCount and prev_index < maxCount then
                local nextIndex = prev_index + 1
                isAutoSkipRun = true
                lua_thread.create(function()
                    wait(300)
                    cmd_vtow(("%d %s %d%s"):format(prev_group or 1, prev_query, nextIndex, prev_fix and " fix" or ""))
                end)
            elseif AUTO_RETRY_MS > 0 then
                lua_thread.create(function()
                    wait(AUTO_RETRY_MS)
                    cmd_vtow(("%d %s %d%s"):format(prev_group or 1, prev_query, prev_index, prev_fix and " fix" or ""))
                end)
            end
        else
            say("{FFAA00}Tow failed for one or more vehicles in use. Flow reset.")
        end
        return false
    end

    if lower:find("selected vehicle is successfully towed", 1, true) then
        if not awaitingRepairDialog then
            local answerYes = (want.repairAnswer == true)
            local perVehicleCost = get_per_vehicle_cost(answerYes)

            if want.isAll then
                local count = tonumber(lastAllTowedCount)
                local total = tonumber(lastAllTowedTotal)
                local totalCost = count and (count * perVehicleCost) or nil
                local base = (count and total) and ("{00FF00}" .. tostring(count) .. "/" .. tostring(total) .. " vehicles")
                    or (count and ("{00FF00}" .. tostring(count) .. " vehicles") or "{00FF00}All vehicles")

                if answerYes then
                    if totalCost then
                        say(("%s {00FF00}have been towed and repaired for {FFFFFF}$%s{00FF00} ({FFFFFF}$%s{00FF00} each).")
                            :format(base, format_money(totalCost), format_money(perVehicleCost)))
                    else
                        say(("%s {00FF00}have been towed and repaired ({FFFFFF}$%s{00FF00} each).")
                            :format(base, format_money(perVehicleCost)))
                    end
                else
                    if totalCost then
                        say(("%s {00FF00}have been towed for {FFFFFF}$%s{00FF00} ({FFFFFF}$%s{00FF00} each).")
                            :format(base, format_money(totalCost), format_money(perVehicleCost)))
                    else
                        say(("%s {00FF00}have been towed ({FFFFFF}$%s{00FF00} each).")
                            :format(base, format_money(perVehicleCost)))
                    end
                end
            else
                if answerYes then
                    say(("{00FF00}Vehicle {FFFFFF}%s (%s) {00FF00}has been towed and repaired for {FFFFFF}$%s{00FF00}."):format(
                        towed_vehicle_info.name, towed_vehicle_info.nickname, format_money(perVehicleCost)))
                else
                    say(("{00FF00}Vehicle {FFFFFF}%s (%s) {00FF00}has been towed for {FFFFFF}$%s{00FF00}."):format(
                        towed_vehicle_info.name, towed_vehicle_info.nickname, format_money(perVehicleCost)))
                end
            end
            reset_flow("tow success message observed")
        end
        return false
    end
end

-- Some servers emit certain lines via the "server message color" event; route through the same filter.
function sampev.onServerMessageColor(color, msg)
    return sampev.onServerMessage(color, msg)
end

-- Main loop
function main()
    if not isSampLoaded() or not isSampfuncsLoaded() then return end
    while true do
        local ok, spawned = pcall(sampIsLocalPlayerSpawned)
        if ok and spawned then break end
        wait(100)
    end

    sampRegisterChatCommand("vtow", cmd_vtow)
    sampRegisterChatCommand("vtowall", cmd_vtowall)
    sampRegisterChatCommand("ptow", cmd_ptow)
    sampRegisterChatCommand("pvtow", cmd_ptow)
    sampRegisterChatCommand("ptowall", cmd_ptowall)
    sampRegisterChatCommand("pvtowall", cmd_ptowall)
    sampRegisterChatCommand("vtowdebug", cmd_vtowdebug)

    debug_print("VTow loaded! /vtow [group] car [index] [fix] | /vtowall [group] [fix] | /ptow car [index] [fix] | /ptow <slot> [fix] | /ptow all [fix] | /ptowall [fix]")

    while true do
        wait(0)
local timeoutMs = (flow == "PERKCHECK") and PERK_CHECK_TIMEOUT_MS or DIALOG_TIMEOUT_MS
if towActive and lastDialogTick > 0 and (now_ms() - lastDialogTick) > timeoutMs then
    if flow == "PERKCHECK" then
        debug_print("Perk check timed out, continuing with default (repair ON)")
        perkCheckTimeoutFallback()
    else
        debug_print("Timeout waiting for next dialog; aborting flow")
        reset_flow("dialog timeout")
    end
end
    end
end