-- MoonLoader: TrackGPS (/tgps)
-- Remembers the last coordinate pair seen in chat (supports many formats) and runs /gps X Y.
--
-- New behaviour:
--  - /tgps always sends /gps first to clear any active checkpoint.
--  - If there is NO active checkpoint, some servers open a GPS menu dialog.
--    When /tgps triggered it, we auto-close that dialog (press "Close") and ignore it.
--  - If you type /gps yourself, the script will NOT auto-close anything.

script_name('TrackGPS')
script_author('ncta')
script_version('2.4.1')

require 'lib.moonloader'

local sampev_ok, sampev = pcall(require, 'lib.samp.events')

local last = { x = nil, y = nil, raw = nil, source = nil }

-- Timestamp when we last captured a coordinate pair (informational, not required for /tgps logic).
local lastCoordAt = 0.0

-- After /tgps sends /gps (and sometimes /gps X Y), some servers print GPS system messages.
-- We suppress those server messages for a short window so only our [TrackGPS] messages show.
local suppressGpsServerMsgUntil = 0.0

-- /tgps toggle logic uses the server reply "GPS marker removed.".
-- If we see it for the /gps we sent, we STOP (do not re-track) until the user runs /tgps again.
local awaitGpsRemovedUntil = 0.0
local awaitGpsRemovedSaw = false

-- If /gps opens the GPS menu (meaning there was no active marker), we record it
-- so /tgps can proceed without waiting the full await window.
local gpsMenuSeenAt = 0.0

-- Prevent accidental double execution (key repeat, double bind, etc.).
local tgpsBusy = false
local lastTgpsInvokeAt = 0.0

-- When /tgps sends plain "/gps" to clear an existing checkpoint, some servers show a GPS menu dialog.
-- We don't want that dialog to steal focus. This flag is only enabled briefly after /tgps.
local closeNextGpsDialogUntil = 0.0
local function markCloseNextGpsDialog(seconds)
    closeNextGpsDialogUntil = os.clock() + (seconds or 2.0)
end

local function markSuppressGpsServerMsgs(seconds)
    suppressGpsServerMsgUntil = os.clock() + (seconds or 3.0)
end

local function startAwaitGpsRemoved(seconds)
    awaitGpsRemovedUntil = os.clock() + (seconds or 1.2)
    awaitGpsRemovedSaw = false
end


-- ===== helpers =====
local function stripColors(s)
    if not s then return '' end
    -- {FFFFFF} and similar
    s = s:gsub('{%x%x%x%x%x%x}', '')
    -- \x1b colors (rare)
    s = s:gsub('\27%[%d+;?%d*m', '')
    return s
end

local function trim(s)
    return (s:gsub('^%s+', ''):gsub('%s+$', ''))
end

local function stripLeadingTimestamp(s)
    -- strips:
    -- [21:19:47] ...
    -- [21:19] ...
    -- [21:19:47] {FFFFFF} ...
    if not s then return '' end
    s = s:gsub('^%s*%[%d%d:%d%d:%d%d%]%s*', '')
    s = s:gsub('^%s*%[%d%d:%d%d%]%s*', '')
    return s
end

local function stripWalkieFrequencies(s)
    if not s or s == '' then return s end

    -- Walkie talkie / radio frequency format often looks like: [2 | 1]
    -- Those are not map coords, so strip them out so they never get parsed as X Y.
    s = s:gsub('%[%s*%d+%s*|%s*%d+%s*%]', ' ')
    s = s:gsub('%(%s*%d+%s*|%s*%d+%s*%)', ' ')
    s = s:gsub('%{%s*%d+%s*|%s*%d+%s*%}', ' ')
    return s
end

local function sanitize(text)
    local s = trim(stripLeadingTimestamp(stripColors(text or '')))
    s = stripWalkieFrequencies(s)
    return trim(s)
end

local function lower(s) return (s or ''):lower() end

local function hasCoordKeywords(s)
    s = lower(s)

    -- Use word-boundary style checks for short keywords (prevents false hits like "deposited" -> "pos").
    if s:find('%f[%a]gps%f[%A]') then return true end
    if s:find('%f[%a]coord%f[%A]') then return true end
    if s:find('%f[%a]coords%f[%A]') then return true end
    if s:find('%f[%a]coordinates%f[%A]') then return true end
    if s:find('%f[%a]location%f[%A]') then return true end
    if s:find('%f[%a]position%f[%A]') then return true end

    -- Label-style coords
    if s:find('[Xx]%s*[:=]') then return true end
    if s:find('[Yy]%s*[:=]') then return true end
    return false
end

local function isJoinLoginLine(s)
    s = lower(s)
    -- common Valrise/SAMP lines
    if s:find('[join]', 1, true) and s:find('logged', 1, true) then return true end
    if s:find('logged in', 1, true) then return true end
    if s:find('logged out', 1, true) then return true end
    if s:find('connected', 1, true) and s:find('server', 1, true) then return true end
    if s:find('disconnected', 1, true) then return true end
    return false
end

local function toInt(n)
    if n == nil then return nil end
    if n >= 0 then return math.floor(n + 0.5) end
    return -math.floor(math.abs(n) + 0.5)
end

local function inWorldRange(x, y)
    if not x or not y then return false end
    return (math.abs(x) <= 4000) and (math.abs(y) <= 4000)
end

local function acceptXY(x, y, context, kw)
    if not inWorldRange(x, y) then return false end

    -- If no coordinate keywords, be stricter to avoid IDs/timestamps/random numbers.
    if not kw then
        -- don't allow zeros from random triples like "0 0 -2323"
        if x == 0 or y == 0 then return false end

        -- reject small pairs that look like times/ids
        if math.abs(x) < 50 and math.abs(y) < 50 then return false end

        -- if both are positive, require both reasonably big
        if x > 0 and y > 0 then
            if math.abs(x) < 200 or math.abs(y) < 200 then return false end
        end

        -- If line contains obvious id patterns like "(92):" and no keywords, don't accept.
        local l = lower(context)
        if l:find('%(%s*%d+%s*%)%s*[:%-]', 1) then
            return false
        end
    end

    return true
end

-- Parse numbers that can include decimals and optional trailing punctuation like ",".
-- Supports: 714, -541 | 714.0 -541.2 | (714.0, -541.2) | [GPS 714 -541]
local function findNumbers(s)
    local nums = {}
    if not s or s == '' then return nums end

    -- Replace commas between numbers with space, but keep minus/plus and decimals.
    -- We want "714, -541" to parse as 714 and -541.
    local t = s:gsub(',', ' ')

    -- Find numeric tokens with optional decimal part.
    for tok in t:gmatch('[+-]?%d+%.?%d*') do
        local n = tonumber(tok)
        if n ~= nil then
            nums[#nums+1] = n
        end
    end
    return nums
end

local function extractCoords(text)
    if not text or text == '' then return nil end
    local raw = stripColors(text)
    local clean = sanitize(raw)
    if clean == '' then return nil end

    -- Hard block join/login noise unless it has explicit coord keywords.
    local kw = hasCoordKeywords(clean)
    if (not kw) and isJoinLoginLine(clean) then return nil end

    -- Priority 1: Explicit X/Y labels
    do
        local x = clean:match('[Xx]%s*[:=]%s*([+-]?%d+%.?%d*)')
        local y = clean:match('[Yy]%s*[:=]%s*([+-]?%d+%.?%d*)')
        if x and y then
            local xi, yi = toInt(tonumber(x)), toInt(tonumber(y))
            if acceptXY(xi, yi, clean, true) then
                return xi, yi, trim(clean)
            end
        end
    end

    -- Priority 2: Keyword then first two numbers after it (gps/coords/pos/location etc)
    do
        local l = lower(clean)
        local kpos = nil
        local keys = { 'gps', 'coords', 'coord', 'coordinates', 'pos', 'position', 'location' }
        for _,k in ipairs(keys) do
            local p = l:find(k, 1, true)
            if p and (not kpos or p < kpos) then kpos = p end
        end
        if kpos then
            local tail = clean:sub(kpos)
            local nums = findNumbers(tail)
            if #nums >= 2 then
                local xi, yi = toInt(nums[1]), toInt(nums[2])
                if acceptXY(xi, yi, clean, true) then
                    return xi, yi, trim(clean)
                end
            end
        end
    end

    -- Priority 3: Bracket/paren formats like "(714.0, -541.2)" or "[714 -541]"
    do
        local inside = clean:match('%b()') or clean:match('%b[]') or clean:match('%b{}')
        if inside then
            local nums = findNumbers(inside)
            if #nums >= 2 then
                local xi, yi = toInt(nums[1]), toInt(nums[2])
                if acceptXY(xi, yi, clean, kw) then
                    return xi, yi, trim(clean)
                end
            end
        end
    end

    -- Priority 4: Plain mission/chat coords. We ONLY accept formats that have a SPACE between numbers.
    -- This prevents money like "$400,000" from being misread as "400 0".
    -- If line contains exactly "X Y Z" (with spaces) -> treat as X Y Z and ALWAYS use 1st+2nd, ignore 3rd.
    do
        -- 3 numbers: X Y Z (ignore Z)
        local ax, ay, az = clean:match('([+-]?%d+%.?%d*)%s*,?%s+([+-]?%d+%.?%d*)%s*,?%s+([+-]?%d+%.?%d*)')
        if ax and ay and az then
            local xi, yi = toInt(tonumber(ax)), toInt(tonumber(ay))
            if acceptXY(xi, yi, clean, kw) then
                return xi, yi, trim(clean)
            end
            return nil
        end

        -- 2 numbers: X Y
        local bx, by = clean:match('([+-]?%d+%.?%d*)%s*,?%s+([+-]?%d+%.?%d*)')
        if bx and by then
            local xi, yi = toInt(tonumber(bx)), toInt(tonumber(by))
            if acceptXY(xi, yi, clean, kw) then
                return xi, yi, trim(clean)
            end
        end
    end

    return nil
end

local function remember(source, text)
    local x, y, raw = extractCoords(text)
    if x and y then
        last.x, last.y, last.raw, last.source = x, y, raw, source
        lastCoordAt = os.clock()
    end
end

local function info(msg)
    sampAddChatMessage(('{00FF90}[TrackGPS]{FFFFFF} %s'):format(msg), -1)
end

function main()
    while not isSampAvailable() do wait(200) end

    sampRegisterChatCommand('tgps', function()
        -- Debounce to prevent accidental double triggers.
        local now = os.clock()
        if tgpsBusy then return end
        if (now - lastTgpsInvokeAt) < 0.35 then return end
        lastTgpsInvokeAt = now
        tgpsBusy = true

        -- Always clear any existing checkpoint first.
        -- If there is none, the server may open the GPS menu: we auto-close it for /tgps.
        markCloseNextGpsDialog(2.0)

        -- Hide noisy GPS system messages that appear because we are running /gps from the script.
        markSuppressGpsServerMsgs(3.0)

        lua_thread.create(function()
            local ok, err = pcall(function()
                -- First command is always /gps.
                -- Your server behaviour: if a marker was active, it replies "GPS marker removed.".
                -- As requested, if that happens we STOP and do not set coords again until the next /tgps.
                startAwaitGpsRemoved(1.8)
                sampSendChat('/gps')

                -- Wait a moment for the server reply.
                local startedAt = os.clock()
                local deadline = awaitGpsRemovedUntil
                while os.clock() <= deadline do
                    if awaitGpsRemovedSaw then
                        info('Cleared GPS.')
                        return
                    end
                    -- If the GPS menu appeared, there was no active marker to remove.
                    if gpsMenuSeenAt >= startedAt then
                        break
                    end
                    wait(40)
                end

                -- No "GPS marker removed." reply observed: set stored coords (if any).
                wait(220)
                if last.x and last.y then
                    -- On your server, if a marker is still considered active, /gps X Y can behave like plain /gps
                    -- and you'll get "GPS marker removed." instead of a new marker.
                    -- If that happens, we do NOT try again in the same /tgps. You must run /tgps again.
                    startAwaitGpsRemoved(0.9)
                    sampSendChat(('/gps %d %d'):format(last.x, last.y))

                    local dl2 = awaitGpsRemovedUntil
                    while os.clock() <= dl2 do
                        if awaitGpsRemovedSaw then
                            info('Cleared GPS.')
                            return
                        end
                        wait(40)
                    end

                    info(('Set GPS to %d %d.'):format(last.x, last.y))
                else
                    info('Cleared GPS (if any). No stored coordinates yet.')
                end
            end)

            tgpsBusy = false
            if not ok then
                info('Error: ' .. tostring(err))
            end
        end)
    end)

sampRegisterChatCommand('tgpslast', function()
        if last.x and last.y then
            info(('Last: %d %d (%s)'):format(last.x, last.y, last.raw or ''))
        else
            info('No GPS stored yet.')
        end
    end)

    if sampev_ok and sampev then
        function sampev.onServerMessage(color, text)
            local clean = sanitize(text or '')
            local lc = lower(clean)

            -- Detect if /tgps just cleared an active GPS marker.
            if os.clock() <= awaitGpsRemovedUntil then
                if lc == 'gps marker removed.' then
                    awaitGpsRemovedSaw = true
                end
            end

            -- Suppress server-side GPS system messages during a short window after /tgps runs.
            if os.clock() <= suppressGpsServerMsgUntil then
                if lc == 'gps marker removed.' then
                    return false
                end
                if lc:find('an icon has been placed on your map', 1, true) then
                    return false
                end
            end

            remember('server', text)
        end

        function sampev.onClientMessage(color, text)
            remember('client', text)
        end

        function sampev.onChatMessage(playerId, text)
            remember('chat', text)
        end

        -- Auto-close the GPS menu that opens when /gps is used without an active checkpoint.
        function sampev.onShowDialog(dialogId, style, title, button1, button2, text)
            if os.clock() > closeNextGpsDialogUntil then
                return true
            end

            local t = lower(stripColors(title or ''))
            local b1 = lower(stripColors(button1 or ''))
            local b2 = lower(stripColors(button2 or ''))
            local body = lower(stripColors(text or ''))

            -- Your screenshot: title = "GPS: main category", buttons = Continue / Close
            -- We want to press the RIGHT button (Close) so it doesn't pick any list item.
            local looksLikeGps = false
            if t:find('gps', 1, true) then looksLikeGps = true end
            if (not looksLikeGps) and (b1:find('gps', 1, true) or b2:find('gps', 1, true)) then looksLikeGps = true end
            if (not looksLikeGps) and (body:find('gps', 1, true) or body:find('checkpoint', 1, true)) then looksLikeGps = true end

            if looksLikeGps then
                -- button = 0 presses the RIGHT button (Close/Cancel)
                sampSendDialogResponse(dialogId, 0, 0, '')
                gpsMenuSeenAt = os.clock()
                closeNextGpsDialogUntil = 0.0
                return false
            end

            return true
        end
    else
        info('Warning: lib.samp.events not found. TrackGPS may not capture chat messages.')
    end

    wait(-1)
end