script_name("TurfListDialog")
script_author("ncta")
script_version("5.5")

local sampev = require 'lib.samp.events'

-- flags
local isWaitingTurfs = false
local isWaitingPerks = false

-- config
local OUR_GROUP = "CHA"

-- storage
local groupTotals = {}
local groupList = {}
local capturable = {}
local underAttack = {}
local ownerIndexByTurf = {}
local perksByTurf = {}
local turfStateByTurf = {}
local groupAlliances = {} -- group -> alliance counts (from /turfs)

-- colors
local PINK = "{FF69B4}"        -- tier indicator color (pink)
local ALLY_BLUE = "{4db5ff}"   -- ally/alliance color (light blue; slightly lighter than STRONG_BLUE)
local ENEMY_COLOR = "{2ECC71}" -- enemy color (green)

-- Derived / UI colors
local OUR_COLOR = ALLY_BLUE      -- perk matching/highlights + allied group lines
local MCF_COLOR = "{7E76DE}"     -- MCF tag color (group name / owner)
local MCF_ALLY_COLOR = ALLY_BLUE -- alliance header / our alliance section color

-- Perk state (re-built each /tlist)
local OUR_ACTIVE_PERKS = {}      -- normalized perk texts active on any OUR_GROUP turf
local PERK_FULL_CONTROL = {}     -- normalized perk texts where ALL turfs with that perk are owned by OUR_GROUP

local STRONG_BLUE = "{0066FF}"   -- strong blue when OUR_GROUP fully controls a perk
local SPECIAL_COLOR = "{B266FF}" -- special label color (e.g., [Special])
local GREY_TIMER = "{6c7887}"    -- cooldown timer color
local GPS_COLOR = "{C0C0C0}"     -- GPS tag color

-- ========== NORMALIZATION ==========
local function strip_colors(s) return (s or ""):gsub("{%x%x%x%x%x%x}", "") end
local function normApos(s) return (s or ""):gsub("‘","'"):gsub("’","'") end
local function normSpacesKeepTabs(s) return (s or ""):gsub("\r",""):gsub("^%s+",""):gsub("%s+$","") end
local function normSpaces(s) return (s or ""):gsub("\r",""):gsub("\t"," "):gsub("%s+"," "):gsub("^%s+",""):gsub("%s+$","") end
local function normName(s)
    s = strip_colors(s or ""); s = normApos(s); s = s:gsub("\t"," "); s = s:lower(); s = normSpaces(s); return s
end

local function eqGroup(a, b)
    return (tostring(a or ""):upper() == tostring(b or ""):upper())
end

local function fmtOwner(owner)
    if eqGroup(owner, OUR_GROUP) then
        return string.format("%s%s{FFFFFF}", MCF_COLOR, owner)
    end
    return owner or ""
end

local function fmtAllianceName(a)
    a = strip_colors(a or "")
    a = normSpaces(a)
    if a == "" then return "" end
    local al = a:lower()
    if al == "solo group" or al == "solo" or al == "none" then
        return "Solo Group"
    end

    -- If already contains a trailing (TAG) or [TAG], normalize to [TAG].
    a = a:gsub("%(([%w%d]+)%)%s*$", "[%1]")
    if a:find("[", 1, true) and a:find("]", 1, true) then
        return a
    end

    -- If it ends with an uppercase tag, turn "Name TAG" into "Name [TAG]"
    local name, tag = a:match("^(.-)%s+([A-Z0-9][A-Z0-9]+)$")
    if name and tag and #tag >= 2 and #tag <= 5 and name:find("%S") then
        return string.format("%s [%s]", name, tag)
    end

    return a
end

-- Extract an alliance TAG from a formatted alliance name.
-- Examples:
--   "Paramount Empire (PE)" -> "PE"
--   "[PE]" -> "PE"
--   "PE" -> "PE"
local function allianceTagOnly(alliance)
    local a = strip_colors(alliance or "")
    a = normSpaces(a)
    if a == "" then return "" end

    -- [TAG] at the end
    local tag = a:match("%[([^%]]+)%]%s*$")
    if tag and tag ~= "" then
        tag = normSpaces(tag)
        -- keep only simple uppercase/numeric tags if possible
        local clean = tag:match("^([A-Z0-9][A-Z0-9]+)$")
        return clean or tag
    end


    -- (TAG) at the end (fallback for server formats)
    tag = a:match("%(([^%)]+)%)%s*$")
    if tag and tag ~= "" then
        tag = normSpaces(tag)
        local clean = tag:match("^([A-Z0-9][A-Z0-9]+)$")
        return clean or tag
    end
    -- [TAG]
    tag = a:match("^%[([^%]]+)%]$")
    if tag and tag ~= "" then
        tag = normSpaces(tag)
        local clean = tag:match("^([A-Z0-9][A-Z0-9]+)$")
        return clean or tag
    end

    -- trailing TAG token
    tag = a:match("([A-Z0-9][A-Z0-9]+)%s*$")
    if tag and #tag >= 2 and #tag <= 6 then
        return tag
    end

    return a
end

local function fmtOwnerAlliance(owner, alliance, allianceColor)
    local own = fmtOwner(owner)
    local a = fmtAllianceName(alliance or "")
    if a == "" then return own end
    local al = a:lower()
    if al == "solo group" or al == "solo" or al == "none" then return own end

    local c = allianceColor or ALLY_BLUE

    -- Avoid double brackets if alliance already includes a "[TAG]" suffix
    if a:find("[", 1, true) and a:find("]", 1, true) then
        return string.format("%s %s%s{FFFFFF}", own, c, a)
    end
    return string.format("%s %s[%s]{FFFFFF}", own, c, a)
end


local function fmtOwnerWithColor(owner, color)
    if eqGroup(owner, OUR_GROUP) then
        local c = color or MCF_COLOR
        return string.format("%s%s{FFFFFF}", c, owner)
    end
    return owner or ""
end

local function fmtOwnerAllianceWithColor(owner, alliance, ownerColor, allianceColor)
    local own = fmtOwnerWithColor(owner, ownerColor)
    local a = fmtAllianceName(alliance or "")
    if a == "" then return own end
    local al = a:lower()
    if al == "solo group" or al == "solo" or al == "none" then return own end

    local c = allianceColor or ALLY_BLUE

    -- Avoid double brackets if alliance already includes a "[TAG]" suffix
    if a:find("[", 1, true) and a:find("]", 1, true) then
        return string.format("%s %s%s{FFFFFF}", own, c, a)
    end
    return string.format("%s %s[%s]{FFFFFF}", own, c, a)
end



-- ========== SPECIAL VEHICLE LABELS ==========
-- Special vehicle allowed in specific turfs
local SPECIAL_TURFS = {
    [normName("Ocean Docks")] = true,
    [normName("Eastern Basin")] = true,
    [normName("Ghost Town")] = true,
    [normName("Willowfield Factory")] = true,
    [normName("Playa De Seville")] = true,
    [normName("Junkyard")] = true,
    [normName("LV Warehouse North")] = true,
}

local function specialTag(turfName)
    if SPECIAL_TURFS[normName(turfName or "")] then
        return string.format(" %s[Special]{FFFFFF}", SPECIAL_COLOR)
    end
    return ""
end

-- ========== HARDCODED TURF GPS ==========
-- Show a GPS tag right after the turf name for specific turfs
local TURF_GPS = {
    [normName("Jizzy's Club")] = "1910",
    [normName("Idlewood Motel")] = "1909",
    [normName("Big Smoke's Hideout")] = "1907",
}

local function gpsTag(turfName)
    local id = TURF_GPS[normName(turfName or "")]
    if id then
        return string.format(" %s[GPS %s]{FFFFFF}", GPS_COLOR, id)
    end
    return ""
end


-- ========== CUSTOM TURF PERKS ==========
-- Hard overrides for specific turfs (used in all sections)
local CUSTOM_TURF_PERKS = {
    [normName("Goldway Gun Factory")] = { text = "Warehouse", score = 1000, tier = "+++++" },
}

-- ========== PERK SCORING ==========
local perkWeights = {
    -- +++ top
    ["repair group vehicles on spawn"] = 95,
    ["discount on fuel and pay 'n' spray"] = 90,
    ["increased smuggle mission rewards"] = 90,
    ["increased drug production grams"] = 85,

    -- ++ medium
    ["drug dealer price bonus"] = 75,
    ["blueberry factory bonus"] = 70,
    ["increased business profits"] = 65,

    -- + low
    ["periodical points for ownership"] = 60,
    ["20% time reduction for revives"] = 60,
    ["increased bonus for thief deliveries"] = 55,
}

local function normalizePerkText(s)
    s = normApos((s or ""):lower())
    s = s:gsub("pay ?[‘’']?n[‘’']? ?spray", "pay 'n' spray")
    s = s:gsub("discount on fuel and pay ?'n'? ?spray", "discount on fuel and pay 'n' spray")
    s = s:gsub("20%%+%s*time reduction for revives", "20% time reduction for revives")
    s = s:gsub("periodic%a* points for ownership", "periodical points for ownership")
    return s
end

local function scorePerk(perkText)
    local t = normalizePerkText(perkText); local best = 0
    for p,w in pairs(perkWeights) do if t:find(p,1,true) and w>best then best=w end end
    local tier = best>=85 and "+++" or (best>=60 and "++" or "+")
    return best, tier
end


local function isValidPerkText(txt)
    if not txt or txt == "" then return false end
    local t = normalizePerkText(txt)
    for phrase, _ in pairs(perkWeights) do
        if t:find(phrase, 1, true) then return true end
    end
    local wc = 0
    for _ in t:gmatch("%a%a%a+") do wc = wc + 1 end
    return wc >= 2
end

-- ========== MAIN ==========
function main()
    repeat wait(0) until isSampAvailable()
    sampRegisterChatCommand("tlist", cmd_tlist)
end

function cmd_tlist()
    groupTotals, groupList = {}, {}
    groupAlliances = {}
    capturable, underAttack = {}, {}
    ownerIndexByTurf, perksByTurf, turfStateByTurf = {}, {}, {}

    -- request turfs first
    isWaitingTurfs = true
    sampSendChat("/turfs")

    -- then request perks after turfs dialog is processed (or after a short timeout)
    lua_thread.create(function()
        local waited = 0
        while isWaitingTurfs and waited < 4000 do
            wait(100)
            waited = waited + 100
        end
        isWaitingPerks = true
        sampSendChat("/turfperks all")
    end)
end

-- ========== TIME ==========
local function parseTimeToSeconds(t)
    if not t then return 0 end
    local tl = (t or ""):lower()

    -- treat NOW and [NOW] (and similar) as 0
    if tl:find("now", 1, true) then
        return 0
    end

    local h = tonumber(tl:match("(%d+)%s*h")) or 0
    local m = tonumber(tl:match("(%d+)%s*m")) or 0
    local s = tonumber(tl:match("(%d+)%s*s")) or 0
    return h*3600 + m*60 + s
end

-- ========== DIALOG ==========
function sampev.onShowDialog(id, style, title, b1, b2, text)
    local titleClean = strip_colors(title or "")
    local tl = titleClean:lower()

    -- Turfs dialog
    if isWaitingTurfs and tl:find("turfs", 1, true) then
        isWaitingTurfs = false
        sampSendDialogResponse(id, 0, 0, "")
        local ok, err = pcall(parseTurfs, text or "")
        if not ok then
            sampAddChatMessage("{FF0000}[tlist] parseTurfs error: {FFFFFF}" .. tostring(err), -1)
        end
        return false
    end

    -- Turf perks dialog
    if isWaitingPerks and (tl:find("turf perks", 1, true) or (tl:find("perks", 1, true) and tl:find("turf", 1, true))) then
        isWaitingPerks = false
        sampSendDialogResponse(id, 0, 0, "")
        local ok1, err1 = pcall(parsePerks, text or "")
        if not ok1 then
            sampAddChatMessage("{FF0000}[tlist] parsePerks error: {FFFFFF}" .. tostring(err1), -1)
            return false
        end
        local ok2, err2 = pcall(renderFinalSummary)
        if not ok2 then
            sampAddChatMessage("{FF0000}[tlist] render error: {FFFFFF}" .. tostring(err2), -1)
        end
        return false
    end
end

-- ========== /TURFS PARSER ==========
local function tokenLooksLikeStatus(tok)
    tok = tostring(tok or "")
    if tok == "" then return true end
    local tl = tok:lower()

    -- common status starters
    if tl == "capturable" or tl == "cooldown" or tl == "locked" then return true end
    if tok:match("^%d") then return true end
    if tok:match("%d+[hms]") then return true end
    if tl:find("now", 1, true) then return true end
    if tok:find("Under", 1, true) or tok:find("Attack", 1, true) then return true end

    -- bracketed/parenthesized tokens: treat as status only if numeric/time-ish (so "(PE)" is NOT a status)
    if tok:match("^[%[%(%{]") then
        local inside = tok:gsub("^[%[%(%{]", ""):gsub("[%]%)%}]", "")
        local il = inside:lower()
        if inside:match("^%d") then return true end
        if inside:match("%d+[hms]") then return true end
        if il:find("now", 1, true) then return true end
        return false
    end

    return false
end

function parseTurfs(text)
    text = strip_colors(text or "")
    for line in text:gmatch("[^\r\n]+") do
        if not line:find("Turf%s+Type") then
            local parts = {}; for part in line:gmatch("%S+") do table.insert(parts, part) end
            local globalIndex; for i,v in ipairs(parts) do if v=="Global" then globalIndex=i break end end
            if globalIndex then
                local turfName = table.concat(parts," ",1,globalIndex-1)
                local ownerName = parts[globalIndex+1] or ""
local allianceName = "Solo Group"

-- Alliance column support (multi-word, supports "(TAG)" tokens):
-- Everything after owner until the status/timer begins is treated as the alliance.
local statusStartIndex = globalIndex + 2
local idx = globalIndex + 2
local allianceTokens = {}

while idx <= #parts and not tokenLooksLikeStatus(parts[idx]) do
    table.insert(allianceTokens, parts[idx])
    idx = idx + 1
end

if #allianceTokens > 0 then
    allianceName = fmtAllianceName(table.concat(allianceTokens, " "))
    statusStartIndex = idx
end

local capturableStatus = table.concat(parts, " ", statusStartIndex)

                -- Some lines end like "- OWNER (ALLIANCE)" => trust tail values.
                local ownerFromTail, allianceFromTail = capturableStatus:match("%-%s*(.-)%s*%((.-)%)%s*$")
                if not ownerFromTail then
                    ownerFromTail, allianceFromTail = capturableStatus:match("%-%s*(.-)%s*%[(.-)%]%s*$")
                end
                if ownerFromTail and ownerFromTail ~= "" then ownerName = normSpaces(ownerFromTail) end
                if allianceFromTail and allianceFromTail ~= "" then allianceName = fmtAllianceName(normSpaces(allianceFromTail)) end

                local groupName = ownerName

                local attackParticipants = capturableStatus:match("Under Attack %((%d+)%)")
                if attackParticipants then
                    table.insert(underAttack,{
                        turf=turfName,
                        owner=groupName,
                        alliance=allianceName,
                        participants=tonumber(attackParticipants),
                        status=capturableStatus
                    })
                end

                groupTotals[groupName] = (groupTotals[groupName] or 0) + 1
                groupAlliances[groupName] = groupAlliances[groupName] or {}
                groupAlliances[groupName][allianceName] = (groupAlliances[groupName][allianceName] or 0) + 1
                ownerIndexByTurf[normName(turfName)] = { raw=turfName, owner=groupName, alliance=allianceName }
                local tsec = parseTimeToSeconds(capturableStatus)
                turfStateByTurf[normName(turfName)] = { status=capturableStatus, time=tsec, underAttack = (attackParticipants ~= nil) }

                -- Capturable list: exclude our own group and exclude "Under Attack"
                if capturableStatus ~= "" and (not eqGroup(groupName, OUR_GROUP)) and (not attackParticipants) then
                    table.insert(capturable,{
                        turf=turfName,
                        owner=groupName,
                        alliance=allianceName,
                        status=capturableStatus,
                        time=tsec
                    })
                end
            end
        end
    end

    groupList = {}
    for name,total in pairs(groupTotals) do
        local primaryAlliance = "Solo Group"
        local counts = groupAlliances[name]
        if counts then
            local bestA, bestC = "Solo Group", 0
            for an, c in pairs(counts) do
                if (c or 0) > bestC then bestA, bestC = an, c end
            end
            primaryAlliance = bestA
        end
        table.insert(groupList, { name=name, total=total, alliance=primaryAlliance })
    end

    table.sort(groupList, function(a,b)
        local A = eqGroup(a.name, OUR_GROUP)
        local B = eqGroup(b.name, OUR_GROUP)
        if A ~= B then return A end
        if a.total ~= b.total then return a.total > b.total end
        return (a.name or ""):lower() < (b.name or ""):lower()
    end)

    table.sort(capturable,function(a,b)
        if a.time ~= b.time then return a.time < b.time end
        if (a.owner or ""):lower() ~= (b.owner or ""):lower() then return (a.owner or ""):lower() < (b.owner or ""):lower() end
        return (a.turf or ""):lower() < (b.turf or ""):lower()
    end)
end

-- ========== PERKS PARSER ==========
local function join_wrapped_lines(lines)
    local joined = {}; local buffer = ""
    for i=1,#lines do
        local line = lines[i]
        if type(line)=="string" and line:find("\t",1,true) then
            if buffer~="" then table.insert(joined, buffer) end
            buffer = line
        else
            if type(line)=="string" then
                if buffer=="" then buffer=line else buffer = buffer .. " " .. line end
            end
        end
    end
    if buffer~="" then
        if buffer:find("\t",1,true) then
            table.insert(joined, buffer)
        else
            local nameGuess, perkGuess = buffer:match("^(.-)%s%s+(.+)$")
            if nameGuess and isValidPerkText(perkGuess) then
                table.insert(joined, nameGuess .. "\t" .. perkGuess)
            end
        end
    end
    return joined
end

local function addPerk(turf, perkText)
    if type(turf)~="string" or type(perkText)~="string" then return end
    if turf=="" or perkText=="" then return end
    if not isValidPerkText(perkText) then return end
    local key = normName(turf)
    perkText = perkText:gsub("(%d+)%s+(%d+)(%a+)","%1%3")
    local score, tier = scorePerk(perkText)
    if score <= 0 then return end
    perksByTurf[key] = perksByTurf[key] or {}
    table.insert(perksByTurf[key], { text=normSpaces(perkText), score=score, tier=tier })
end

function parsePerks(text)
    text = strip_colors(text or "")
    local rawLines = {}
    for raw in text:gmatch("[^\r\n]+") do
        local line = normApos(raw); if type(line)=="string" then table.insert(rawLines, line) end
    end
    local lines = join_wrapped_lines(rawLines)
    for i=1,#lines do
        local raw = lines[i]
        if type(raw)=="string" then
            local nTab, pTab = raw:match("^(.-)\t(.+)$")
            if nTab and pTab then
                addPerk(normSpacesKeepTabs(nTab), normSpacesKeepTabs(pTab))
            else
                local nSp, pSp = raw:match("^(.-)%s%s+(.+)$")
                if nSp and pSp then
                    addPerk(normSpaces(nSp), normSpaces(pSp))
                else
                    local name1, perks1 = raw:match("^(.-)%s*%-%s*(.+)$")
                    if name1 and perks1 then
                        for perk in perks1:gmatch("([^,]+)") do addPerk(name1, normSpaces(perk)) end
                    else
                        local name2, perks2 = raw:match("^(.-)%s*:%s*(.+)$")
                        if name2 and perks2 then
                            for perk in perks2:gmatch("([^,]+)") do addPerk(name2, normSpaces(perk)) end
                        end
                    end
                end
            end
        end
    end
end

-- ========== MATCHING & RENDER ==========
local function pickBestPerk(key)
    local custom = CUSTOM_TURF_PERKS[key]
    if custom then return custom end
    local list = perksByTurf[key]
    if not list or #list==0 then return nil end
    table.sort(list,function(a,b) return a.score>b.score end)
    return list[1]
end

local function rebuildOurActivePerks()
    OUR_ACTIVE_PERKS = {}
    PERK_FULL_CONTROL = {}

    -- Track: for each normalized perk text, how many turfs have it in total,
    -- and how many of those are owned by OUR_GROUP.
    --
    -- IMPORTANT: use pickBestPerk() so CUSTOM_TURF_PERKS are included.
    -- Otherwise singular perks (like Warehouse) never count as "fully controlled"
    -- and stay light-blue even when owned.
    local perkCounts = {}

    for turfKey, meta in pairs(ownerIndexByTurf) do
        local best = pickBestPerk(turfKey)
        if best and best.text and best.text ~= "" then
            local perkKey = normalizePerkText(best.text)
            perkCounts[perkKey] = perkCounts[perkKey] or { total = 0, ours = 0 }
            perkCounts[perkKey].total = perkCounts[perkKey].total + 1

            if meta and eqGroup(meta.owner, OUR_GROUP) then
                perkCounts[perkKey].ours = perkCounts[perkKey].ours + 1
                OUR_ACTIVE_PERKS[perkKey] = true
            end
        end
    end

    -- A perk is "fully controlled" if every turf that has it is owned by OUR_GROUP.
    for perkKey, c in pairs(perkCounts) do
        if (c.total or 0) > 0 and (c.ours or 0) == (c.total or 0) then
            PERK_FULL_CONTROL[perkKey] = true
        end
    end
end

function renderFinalSummary()
    rebuildOurActivePerks()
    local output = ""

    -- Groups overview grouped by alliances
    local function isSoloAlliance(a)
        a = strip_colors(a or "")
        local al = a:lower()
        return a == "" or al == "solo group" or al == "solo" or al == "none"
    end

    local mcfAlliance = ""
    local mcfName = OUR_GROUP
    local mcfTotal = 0

    for _, g in ipairs(groupList) do
        local gName = strip_colors(g.name or "")
        if eqGroup(gName, OUR_GROUP) then
            mcfName = gName
            mcfTotal = g.total or 0
            if not isSoloAlliance(g.alliance) then
                mcfAlliance = strip_colors(g.alliance or "")
            end
            break
        end
    end

    local ourInAlliance = (mcfAlliance ~= "")

    -- If we're in an alliance, do NOT list allied turfs as capturable/cooldown targets.
    -- This fixes cases where MCF-alliance owned turfs were incorrectly shown under "Capturable turfs".
    local function sameAlliance(a, b)
        local ak = allianceTagOnly(fmtAllianceName(a or ""))
        local bk = allianceTagOnly(fmtAllianceName(b or ""))
        ak = tostring(ak or ""):upper()
        bk = tostring(bk or ""):upper()
        return ak ~= "" and bk ~= "" and ak == bk
    end

    -- Color rule: use ally color for OUR_GROUP / allied items; enemies use ENEMY_COLOR.
    local function relationColor(owner, alliance)
        if eqGroup(owner, OUR_GROUP) then return ALLY_BLUE end
        if ourInAlliance and sameAlliance(alliance, mcfAlliance) then return ALLY_BLUE end
        return ENEMY_COLOR
    end

    local visibleCapturable = {}
    if ourInAlliance then
        for _, t in ipairs(capturable) do
            if not sameAlliance(t.alliance, mcfAlliance) then
                table.insert(visibleCapturable, t)
            end
        end
    else
        visibleCapturable = capturable
    end

    local alliances = {}
    local soloGroups = {}

    for _, g in ipairs(groupList) do
        local aName = strip_colors(g.alliance or "")
        if isSoloAlliance(aName) then
            table.insert(soloGroups, g)
        else
            alliances[aName] = alliances[aName] or {}
            table.insert(alliances[aName], g)
        end
    end

    local function sortByTotalDesc(list)
        table.sort(list, function(a,b)
            local at = a.total or 0
            local bt = b.total or 0
            if at ~= bt then return at > bt end
            return (strip_colors(a.name or ""):lower()) < (strip_colors(b.name or ""):lower())
        end)
    end

    -- 1) MCF alliance first (or MCF alone if not allied)
    if mcfAlliance ~= "" and alliances[mcfAlliance] then
        output = output .. string.format("%s[%s]{FFFFFF}\n", ALLY_BLUE, fmtAllianceName(mcfAlliance))

        local list = alliances[mcfAlliance]
        sortByTotalDesc(list)

        -- Ensure MCF is always first in its alliance
        table.sort(list, function(a,b)
            local A = eqGroup(strip_colors(a.name or ""), OUR_GROUP)
            local B = eqGroup(strip_colors(b.name or ""), OUR_GROUP)
            if A ~= B then return A end
            local at = a.total or 0
            local bt = b.total or 0
            if at ~= bt then return at > bt end
            return (strip_colors(a.name or ""):lower()) < (strip_colors(b.name or ""):lower())
        end)

        for _, g in ipairs(list) do
            local gName = strip_colors(g.name or "")
            local gTotal = g.total or 0
            local gColor = eqGroup(gName, OUR_GROUP) and MCF_COLOR or OUR_COLOR
            output = output .. string.format("{FFFFFF}-> %s%s{FFFFFF} (%d turfs)\n", gColor, gName, gTotal)
        end
        output = output .. "\n"

        -- Remove it so it won't print again in "other alliances"
        alliances[mcfAlliance] = nil
    else
        -- MCF alone (no alliance)
        output = output .. string.format("%s[%s]{FFFFFF}\n", ALLY_BLUE, mcfName)
        output = output .. string.format("{FFFFFF}-> %s%s{FFFFFF} (%d turfs)\n\n", MCF_COLOR, mcfName, mcfTotal)
    end

    -- 2) Other alliances (enemy alliances)
    local allianceNames = {}
    for aName, _ in pairs(alliances) do
        table.insert(allianceNames, aName)
    end
    table.sort(allianceNames, function(a,b) return tostring(a):lower() < tostring(b):lower() end)

    for _, an in ipairs(allianceNames) do
        output = output .. string.format("%s[%s]{FFFFFF}\n", ENEMY_COLOR, fmtAllianceName(an))

        local list = alliances[an]
        sortByTotalDesc(list)

        for _, g in ipairs(list) do
            local gName = strip_colors(g.name or "")
            output = output .. string.format("{FFFFFF}-> %s%s{FFFFFF} (%d turfs)\n", ENEMY_COLOR, gName, g.total or 0)
        end
        output = output .. "\n"
    end

    -- 3) Non-allied groups
    local filteredSolo = {}
    for _, g in ipairs(soloGroups) do
        local gName = strip_colors(g.name or "")
        if not eqGroup(gName, OUR_GROUP) then
            table.insert(filteredSolo, g)
        end
    end

    if #filteredSolo > 0 then
        output = output .. string.format("%s[Non-Allied Groups]{FFFFFF}\n", ENEMY_COLOR)

        sortByTotalDesc(filteredSolo)

        for _, g in ipairs(filteredSolo) do
            local gName = strip_colors(g.name or "")
            output = output .. string.format("{FFFFFF}-> %s%s{FFFFFF} (%d turfs)\n", ENEMY_COLOR, gName, g.total or 0)
        end
        output = output .. "\n"
    end

-- Under Attack: shows tier + perk name; highlighted if matching OUR_GROUP perk
    if #underAttack > 0 then
        table.sort(underAttack, function(a,b)
            local A = eqGroup(a.owner, OUR_GROUP)
            local B = eqGroup(b.owner, OUR_GROUP)
            if A ~= B then return A end
            if (a.participants or 0) ~= (b.participants or 0) then return (a.participants or 0) > (b.participants or 0) end
            return (a.turf or ""):lower() < (b.turf or ""):lower()
        end)

        output = output .. "{FF0000}Turfs Under Attack:{FFFFFF}\n"
        for _, t in ipairs(underAttack) do
            local key = normName(t.turf)
            local best = pickBestPerk(key)
            local rel = relationColor(t.owner, t.alliance)
            local ownerFmt = fmtOwnerAlliance(t.owner, t.alliance, rel)

            if best then
                local perkKey = normalizePerkText(best.text)
                local colorForTier = OUR_ACTIVE_PERKS[perkKey] and OUR_COLOR or PINK

                if eqGroup(t.owner, OUR_GROUP) and PERK_FULL_CONTROL[perkKey] then
                    colorForTier = STRONG_BLUE
                end
                local tierText = best.tier
                local tierColored = string.format("%s[%s]{FFFFFF}", colorForTier, tierText)
                output = output .. string.format(
                    "{FFFFFF}%s%s%s - %s%s {FF0000}[%d participants] {FFFFFF}%s [%s]\n",
                    t.turf, gpsTag(t.turf), specialTag(t.turf), rel, ownerFmt, t.participants, tierColored, best.text
                )
            else
                output = output .. string.format(
                    "{FFFFFF}%s%s%s - %s%s {FF0000}[%d participants]{FFFFFF}\n",
                    t.turf, gpsTag(t.turf), specialTag(t.turf), rel, ownerFmt, t.participants
                )
            end
        end
        output = output .. "\n"
    end

    -- Split capturable into NOW vs cooldown
    local capturableNow, cooldownTurfs = {}, {}
    for _, t in ipairs(visibleCapturable) do
        if (t.time or 0) <= 0 then
            table.insert(capturableNow, t)
        else
            table.insert(cooldownTurfs, t)
        end
    end

    -- Sort: owner -> name
    table.sort(capturableNow, function(a,b)
        local aa = (a.owner or ""):lower()
        local ba = (b.owner or ""):lower()
        if aa ~= ba then return aa < ba end
        return (a.turf or ""):lower() < (b.turf or ""):lower()
    end)

    table.sort(cooldownTurfs, function(a,b)
        local at = (a.time or 0)
        local bt = (b.time or 0)
        if at ~= bt then return at < bt end

        local aa = (a.owner or ""):lower()
        local ba = (b.owner or ""):lower()
        if aa ~= ba then return aa < ba end

        return (a.turf or ""):lower() < (b.turf or ""):lower()
    end)

    -- Capturable turfs (NOW)
    if #capturableNow > 0 then
        output = output .. "{FFFF00}Capturable turfs:\n"
        for _, t in ipairs(capturableNow) do
            local key = normName(t.turf)
            local best = pickBestPerk(key)
            local rel = relationColor(t.owner, t.alliance)
            local ownerFmt = fmtOwnerAlliance(t.owner, t.alliance, rel)

            if best then
                local perkKey = normalizePerkText(best.text)
                local colorForTier = OUR_ACTIVE_PERKS[perkKey] and OUR_COLOR or PINK

                if eqGroup(t.owner, OUR_GROUP) and PERK_FULL_CONTROL[perkKey] then
                    colorForTier = STRONG_BLUE
                end
                local tierText = best.tier
                local tierColored = string.format("%s[%s]{FFFFFF}", colorForTier, tierText)
                output = output .. string.format(
                    "{FFFFFF}%s%s%s - %s%s {FFFFFF}%s [%s]\n",
                    t.turf, gpsTag(t.turf), specialTag(t.turf), rel, ownerFmt, tierColored, best.text
                )
            else
                output = output .. string.format(
                    "{FFFFFF}%s%s%s - %s%s{FFFFFF}\n",
                    t.turf, gpsTag(t.turf), specialTag(t.turf), rel, ownerFmt
                )
            end
        end
        output = output .. "\n"
    end

    -- Cooldown turfs (timer)
    if #cooldownTurfs > 0 then
        output = output .. "{FFA500}Cooldown turfs:\n"
        for _, t in ipairs(cooldownTurfs) do
            local key = normName(t.turf)
            local best = pickBestPerk(key)
            local rel = relationColor(t.owner, t.alliance)
            local ownerFmt = fmtOwnerAlliance(t.owner, t.alliance, rel)

            if best then
                local perkKey = normalizePerkText(best.text)
                local colorForTier = OUR_ACTIVE_PERKS[perkKey] and OUR_COLOR or PINK

                if eqGroup(t.owner, OUR_GROUP) and PERK_FULL_CONTROL[perkKey] then
                    colorForTier = STRONG_BLUE
                end
                local tierText = best.tier
                local tierColored = string.format("%s[%s]{FFFFFF}", colorForTier, tierText)
                output = output .. string.format(
                    "{FFFFFF}%s%s%s - %s%s %s[%s]{FFFFFF} %s [%s]\n",
                    t.turf, gpsTag(t.turf), specialTag(t.turf), rel, ownerFmt, GREY_TIMER, t.status, tierColored, best.text
                )
            else
                output = output .. string.format(
                    "{FFFFFF}%s%s%s - %s%s %s[%s]{FFFFFF}\n",
                    t.turf, gpsTag(t.turf), specialTag(t.turf), rel, ownerFmt, GREY_TIMER, t.status
                )
            end
        end
        output = output .. "\n"
    end

    -- Footer: OUR_GROUP / ALLIANCE turf tiers
    local ourCapturable, ourCooldown, ourUnderAttack = {}, {}, {}
    for key, meta in pairs(ownerIndexByTurf) do
        local isOurs = eqGroup(meta.owner, OUR_GROUP)
        local isAllied = false
        if ourInAlliance then
            isAllied = sameAlliance(meta.alliance, mcfAlliance)
        end

        -- If we're in an alliance, include allied turfs (same alliance tag) in our sections.
        if isOurs or isAllied then
            local best = pickBestPerk(key)
            local state = turfStateByTurf[key]
            local turfNameFmt = "{FFFFFF}" .. (meta.raw or "")
            local timeTag = ""
            if state and (not state.underAttack) and (state.time or 0) > 0 then
                timeTag = string.format(" %s[%s]{FFFFFF}", GREY_TIMER, (state.status or ""))
            end

            -- In alliance sections: color MCF + allies, but don't append alliance text.
            local ownerColor = isOurs and MCF_COLOR or OUR_COLOR
            local ownerFmt = fmtOwnerWithColor(meta.owner, ownerColor)

            local line = ""
            if best then
                local perkKey = normalizePerkText(best.text)
                local tierColor = OUR_COLOR

                if PERK_FULL_CONTROL[perkKey] then tierColor = STRONG_BLUE end
                local tierText = best.tier
                line = string.format(
                    "%s%s%s - %s%s%s %s[%s]{FFFFFF} [%s]",
                    turfNameFmt, gpsTag(meta.raw), specialTag(meta.raw), ALLY_BLUE, ownerFmt, timeTag, tierColor, tierText, best.text
                )
            else
                line = string.format(
                    "%s%s%s - %s%s%s{FFFFFF}",
                    turfNameFmt, gpsTag(meta.raw), specialTag(meta.raw), ALLY_BLUE, ownerFmt, timeTag
                )
            end

            -- Sort: capturable first, then cooldown (least -> most), then under attack last
            local sortBucket = 0
            local sortTime = 0
            local sortName = (meta.raw or ""):lower()
            if state and state.underAttack then
                sortBucket = 2
                sortTime = 0
            else
                local tsec = (state and state.time) or 0
                if tsec > 0 then
                    sortBucket = 1
                    sortTime = tsec
                else
                    sortBucket = 0
                    sortTime = 0
                end
            end

            if sortBucket == 2 then
                table.insert(ourUnderAttack, { name = sortName, line = line })
            elseif sortBucket == 1 then
                table.insert(ourCooldown, { time = sortTime, name = sortName, line = line })
            else
                table.insert(ourCapturable, { name = sortName, line = line })
            end
        end
    end

    table.sort(ourCapturable, function(a,b) return a.name < b.name end)
    table.sort(ourCooldown, function(a,b)
        if a.time ~= b.time then return a.time < b.time end
        return a.name < b.name
    end)
    table.sort(ourUnderAttack, function(a,b) return a.name < b.name end)

    local function appendLines(list)
        for _, e in ipairs(list) do
            output = output .. e.line .. "\n"
        end
    end

    if (#ourCapturable + #ourCooldown + #ourUnderAttack) > 0 then
        -- Header for OUR_GROUP sections:
        -- If we're in an alliance, show only the TAG: "TAG's available turfs:".
        -- Otherwise show "MCF available turfs:".
        local headerText = OUR_GROUP
        local headerColor = MCF_COLOR
        if ourInAlliance then
            local tag = allianceTagOnly(mcfAlliance)
            local tl = (tag or ""):lower()
            if tag ~= "" and tl ~= "solo group" and tl ~= "solo" and tl ~= "none" then
                headerText = tag .. "'s"
                headerColor = MCF_ALLY_COLOR
            end
        end

        local prefixColored = headerColor .. headerText

        -- Available (capturable) first
        if #ourCapturable > 0 then
            output = output .. string.format("%s available turfs:{FFFFFF}\n", prefixColored)
            appendLines(ourCapturable)
            output = output .. "\n"
        end

        -- Cooldown second
        if #ourCooldown > 0 then
            output = output .. string.format("%s cooldown turfs:{FFFFFF}\n", prefixColored)
            appendLines(ourCooldown)
            output = output .. "\n"
        end

        -- Under attack last (only if any)
        if #ourUnderAttack > 0 then
            output = output .. string.format("%s under attack turfs:{FFFFFF}\n", prefixColored)
            appendLines(ourUnderAttack)
        end
    end

    sampShowDialog(1234, "{FFFFFF}Turf Summary", output, "Close", "", 0)
end