-- json.lua (version 3.0)
--
-- Public domain, by Rici Lake and Nils M Holm.
--
-- This is a simple, robust, and reasonably fast JSON
-- encoder/decoder for Lua. It is compatible with Lua 5.1, 5.2,
-- and 5.3, and with LuaJIT.
--
-- See http://www.json.org/ for the JSON specification.
-- See http://www.lua.org/ for the Lua specification.

local type, tonumber, pairs, error, getmetatable, setmetatable =
      type, tonumber, pairs, error, getmetatable, setmetatable
local select, rawget, rawset = select, rawget, rawset
local floor, huge = math.floor, math.huge
local strrep, strsub, strbyte, strchar, strfind, strmatch, strlen, strformat =
      string.rep, string.sub, string.byte, string.char, string.find, string.match, string.len, string.format

local _ENV = _ENV or {}
local json = { _version = "3.0" }

local escapable = {
   [ "\\" ] = "\\\\",
   [ "\"" ] = "\\\"",
   [ "\b" ] = "\\b",
   [ "\f" ] = "\\f",
   [ "\n" ] = "\\n",
   [ "\r" ] = "\\r",
   [ "\t" ] = "\\t"
}

local function is_key(s)
   return type(s) == "string" and strfind(s, "^[a-zA-Z_][a-zA-Z0-9_]*$")
end

local function is_array(t)
   local n = 0
   if type(t) == "table" then
      for _ in pairs(t) do
         n = n + 1
      end
      for i = 1, n do
         if t[i] == nil then return false end
      end
   end
   return n > 0
end

local function encode_string(s)
   return "\"" .. s:gsub('[\\"\b\f\n\r\t]', escapable)
                  :gsub('[\000-\031]', function(c)
                       return strformat("\\u%04x", strbyte(c))
                    end) .. "\""
end

local function encode_table(t, indent)
   local new_indent = indent and indent .. "   "
   local O, C, D, S = "{", "}", ":", ","
   if indent then
      O, C, D, S = "{\n", "\n}", ": ", ",\n"
   end
   local s = O
   local sep = ""
   if is_array(t) then
      for i = 1, #t do
         s = s .. sep .. new_indent .. json.encode(t[i], new_indent)
         sep = S
      end
   else
      for k, v in pairs(t) do
         if is_key(k) then
            s = s .. sep .. new_indent .. k .. D .. json.encode(v, new_indent)
         else
            s = s .. sep .. new_indent .. encode_string(k) .. D .. json.encode(v, new_indent)
         end
         sep = S
      end
   end
   return s .. C
end

function json.encode(o, indent)
   if o == nil then
      return "null"
   elseif type(o) == "boolean" then
      return o and "true" or "false"
   elseif type(o) == "number" then
      if o ~= o or o == huge or o == -huge then
         return "null"
      else
         return o
      end
   elseif type(o) == "string" then
      return encode_string(o)
   elseif type(o) == "table" then
      return encode_table(o, indent)
   else
      error("cannot encode " .. type(o))
   end
end

local function next_char(s, i, patt)
   i = strfind(s, patt, i)
   if i then
      return i, strsub(s, i, i)
   end
end

local function decode_error(s, i, msg)
   local line, col = 1, 1
   for j = 1, i - 1 do
      if strbyte(s, j) == 10 then
         line = line + 1
         col = 1
      else
         col = col + 1
      end
   end
   error(strformat("%s at line %d, col %d (char %d)", msg, line, col, i))
end

local function codepoint_to_utf8(n)
   if n < 128 then
      return strchar(n)
   elseif n < 2048 then
      return strchar(192 + floor(n / 64),
                     128 + n % 64)
   elseif n < 65536 then
      return strchar(224 + floor(n / 4096),
                     128 + floor(n / 64) % 64,
                     128 + n % 64)
   else
      return strchar(240 + floor(n / 262144),
                     128 + floor(n / 4096) % 64,
                     128 + floor(n / 64) % 64,
                     128 + n % 64)
   end
end

local escapes = {
   b = "\b",
   f = "\f",
   n = "\n",
   r = "\r",
   t = "\t"
}

function json.decode(s)
   local i, c = next_char(s, 1, "[^%s]")
   if not i then return nil, "empty string" end

   local function decode_value()
      i, c = next_char(s, i, "[^%s]")
      if not c then decode_error(s, strlen(s) + 1, "unexpected end of string") end
      if c == "n" then
         if strsub(s, i, i + 3) == "null" then
            i = i + 4
            return nil
         end
      elseif c == "t" then
         if strsub(s, i, i + 3) == "true" then
            i = i + 4
            return true
         end
      elseif c == "f" then
         if strsub(s, i, i + 4) == "false" then
            i = i + 5
            return false
         end
      elseif c == "{" then
         i = i + 1
         local obj = {}
         i, c = next_char(s, i, "[^%s]")
         if c ~= "}" then
            while true do
               local key = decode_value()
               if type(key) ~= "string" then
                  decode_error(s, i, "object key must be a string")
               end
               i, c = next_char(s, i, "[^%s]")
               if c ~= ":" then decode_error(s, i, "expected ':'") end
               obj[key] = decode_value()
               i, c = next_char(s, i, "[^%s]")
               if c == "}" then break end
               if c ~= "," then decode_error(s, i, "expected '}' or ','") end
               i = i + 1
            end
         end
         i = i + 1
         return obj
      elseif c == "[" then
         i = i + 1
         local arr = {}
         i, c = next_char(s, i, "[^%s]")
         if c ~= "]" then
            while true do
               arr[#arr + 1] = decode_value()
               i, c = next_char(s, i, "[^%s]")
               if c == "]" then break end
               if c ~= "," then decode_error(s, i, "expected ']' or ','") end
               i = i + 1
            end
         end
         i = i + 1
         return arr
      elseif c == "\"" then
         i = i + 1
         local j = i
         local res = ""
         while true do
            local c2 = strsub(s, j, j)
            if c2 == "\\" then
               res = res .. strsub(s, i, j - 1)
               local c3 = strsub(s, j + 1, j + 1)
               if c3 == "u" then
                  local hex = strmatch(strsub(s, j + 2, j + 5), "%x%x%x%x")
                  if not hex then decode_error(s, j, "invalid universal character") end
                  res = res .. codepoint_to_utf8(tonumber(hex, 16))
                  j = j + 5
               else
                  res = res .. (escapes[c3] or c3)
                  j = j + 1
               end
               i = j + 1
            elseif c2 == "\"" then
               res = res .. strsub(s, i, j - 1)
               i = j + 1
               return res
            end
            j = j + 1
         end
      else
         local j, n
         j, n = strmatch(s, "^[-+]?[0-9]*.?[0-9]+[eE]?[-+]?[0-9]+", i)
         if j then
            i = j
            return tonumber(n)
         end
      end
      decode_error(s, i, "unexpected character")
   end

   local val = decode_value()
   i, c = next_char(s, i, "[^%s]")
   if c then decode_error(s, i, "trailing garbage") end
   return val
end

return json