script_name("Keybinds Text Manager")
script_author("ncta")
script_version("4.6")

require "lib.moonloader"

-- Optional: enable sampfuncs so we can process other MoonLoader chat commands (e.g. /tgps)
pcall(require, "lib.sampfuncs")
local bit = require "bit"
local inicfg = require "inicfg"

local band, bor = bit.band, bit.bor

local CFG_NAME = "keybinds_cfg"
local BIND_TXT_PATH = getWorkingDirectory() .. "\\config\\" .. CFG_NAME .. ".txt"

-- Modifiers (generic)
local VK_ALT   = 0x12
local VK_CTRL  = 0x11
local VK_SHIFT = 0x10

-- Modifiers (left/right variants) - we block these as "main key" too
local VK_LSHIFT   = 0xA0
local VK_RSHIFT   = 0xA1
local VK_LCONTROL = 0xA2
local VK_RCONTROL = 0xA3
local VK_LMENU    = 0xA4
local VK_RMENU    = 0xA5

-- Mouse buttons
local VK_LBUTTON  = 0x01
local VK_RBUTTON  = 0x02
local VK_MBUTTON  = 0x04
local VK_XBUTTON1 = 0x05
local VK_XBUTTON2 = 0x06

-- Numpad
local VK_NUMPAD0   = 0x60
local VK_NUMPAD1   = 0x61
local VK_NUMPAD2   = 0x62
local VK_NUMPAD3   = 0x63
local VK_NUMPAD4   = 0x64
local VK_NUMPAD5   = 0x65
local VK_NUMPAD6   = 0x66
local VK_NUMPAD7   = 0x67
local VK_NUMPAD8   = 0x68
local VK_NUMPAD9   = 0x69
local VK_MULTIPLY  = 0x6A
local VK_ADD       = 0x6B
local VK_SUBTRACT  = 0x6D
local VK_DECIMAL   = 0x6E
local VK_DIVIDE    = 0x6F

-- Common/system keys
local VK_BACK      = 0x08
local VK_PAUSE     = 0x13
local VK_CAPITAL   = 0x14 -- caps lock
local VK_NUMLOCK   = 0x90
local VK_SCROLL    = 0x91
local VK_SNAPSHOT  = 0x2C -- print screen
local VK_LWIN      = 0x5B
local VK_RWIN      = 0x5C
local VK_APPS      = 0x5D
local VK_CLEAR     = 0x0C -- "Clear" (often numpad 5 when NumLock is off)

-- OEM / punctuation keys
local VK_OEM_1      = 0xBA -- ; :
local VK_OEM_PLUS   = 0xBB -- = +
local VK_OEM_COMMA  = 0xBC -- , <
local VK_OEM_MINUS  = 0xBD -- - _
local VK_OEM_PERIOD = 0xBE -- . >
local VK_OEM_2      = 0xBF -- / ?
local VK_OEM_3      = 0xC0 -- ` ~
local VK_OEM_4      = 0xDB -- [ {
local VK_OEM_5      = 0xDC -- \ |
local VK_OEM_6      = 0xDD -- ] }
local VK_OEM_7      = 0xDE -- ' "

local MOD_ALT   = 1
local MOD_CTRL  = 2
local MOD_SHIFT = 4

local cfgDefault = {
  settings = { block_when_input = 1 },
  meta = { count = 0 },
  items = {}
}

local cfg = inicfg.load(cfgDefault, CFG_NAME)
if not cfg then
  cfg = cfgDefault
  inicfg.save(cfg, CFG_NAME)
end

local function saveIni()
  inicfg.save(cfg, CFG_NAME)
end

local function trim(s) return (tostring(s or ""):gsub("^%s+",""):gsub("%s+$","")) end
local function upperNoSpaces(s)
  s = trim(s):gsub("%s+","")
  return s:upper()
end

local function isInputBlocked()
  if tonumber(cfg.settings.block_when_input) == 0 then return false end
  if type(sampIsChatInputActive) == "function" and sampIsChatInputActive() then return true end
  if type(sampIsDialogActive) == "function" and sampIsDialogActive() then return true end
  if type(isSampfuncsConsoleActive) == "function" and isSampfuncsConsoleActive() then return true end
  if type(sampIsScoreboardOpen) == "function" and sampIsScoreboardOpen() then return true end
  return false
end

local function vkToLabel(vk)
  if vk >= 0x30 and vk <= 0x39 then return string.char(vk) end
  if vk >= 0x41 and vk <= 0x5A then return string.char(vk) end
  if vk >= 0x70 and vk <= 0x87 then return "F" .. tostring(vk - 0x6F) end -- F1..F24

  if vk >= 0x60 and vk <= 0x69 then return "NUMPAD" .. tostring(vk - 0x60) end
  if vk == VK_MULTIPLY then return "NUMPAD*" end
  if vk == VK_ADD then return "NUMPAD+" end
  if vk == VK_SUBTRACT then return "NUMPAD-" end
  if vk == VK_DECIMAL then return "NUMPAD." end
  if vk == VK_DIVIDE then return "NUMPAD/" end

  if vk == VK_LBUTTON then return "MOUSE1" end
  if vk == VK_RBUTTON then return "MOUSE2" end
  if vk == VK_MBUTTON then return "MOUSE3" end
  if vk == VK_XBUTTON1 then return "MOUSE4" end
  if vk == VK_XBUTTON2 then return "MOUSE5" end

  -- punctuation (show as character)
  local oemLabel = {
    [VK_OEM_1] = ";",
    [VK_OEM_PLUS] = "=",
    [VK_OEM_COMMA] = ",",
    [VK_OEM_MINUS] = "-",
    [VK_OEM_PERIOD] = ".",
    [VK_OEM_2] = "/",
    [VK_OEM_3] = "`",
    [VK_OEM_4] = "[",
    [VK_OEM_5] = "\\",
    [VK_OEM_6] = "]",
    [VK_OEM_7] = "'",
  }
  if oemLabel[vk] then return oemLabel[vk] end

  local map = {
    [0x1B]="ESC",[0x20]="SPACE",[0x09]="TAB",[0x0D]="ENTER",
    [0x25]="LEFT",[0x26]="UP",[0x27]="RIGHT",[0x28]="DOWN",
    [0x21]="PGUP",[0x22]="PGDN",[0x23]="END",[0x24]="HOME",
    [0x2D]="INS",[0x2E]="DEL",
    [VK_BACK]="BACKSPACE",
    [VK_CAPITAL]="CAPSLOCK",
    [VK_NUMLOCK]="NUMLOCK",
    [VK_SCROLL]="SCROLLLOCK",
    [VK_SNAPSHOT]="PRTSC",
    [VK_PAUSE]="PAUSE",
    [VK_LWIN]="LWIN",
    [VK_RWIN]="RWIN",
    [VK_APPS]="APPS",
    [VK_CLEAR]="CLEAR",
  }
  return map[vk] or string.format("VK(0x%02X)", vk)
end

local function modPrefix(mod)
  local parts = {}
  if band(mod, MOD_ALT) ~= 0 then parts[#parts+1] = "ALT" end
  if band(mod, MOD_CTRL) ~= 0 then parts[#parts+1] = "CTRL" end
  if band(mod, MOD_SHIFT) ~= 0 then parts[#parts+1] = "SHIFT" end
  if #parts == 0 then return "" end
  return table.concat(parts, "+") .. "+"
end

local function keyDisplay(vk, mod) return modPrefix(mod) .. vkToLabel(vk) end

local function modifiersHeld(mod)
  if band(mod, MOD_ALT) ~= 0 and not isKeyDown(VK_ALT) then return false end
  if band(mod, MOD_CTRL) ~= 0 and not isKeyDown(VK_CTRL) then return false end
  if band(mod, MOD_SHIFT) ~= 0 and not isKeyDown(VK_SHIFT) then return false end
  return true
end

local function isBindPressed(vk, mod) return modifiersHeld(mod) and wasKeyPressed(vk) end

-- NOTE:
--  - '+' is the separator between parts of the key spec.
--  - To bind PLUS/EQUALS key, use: PLUS or EQUALS (don't type '+').
--  - To bind NUMPAD plus, you can type: NUMPAD+ OR NUMPADPLUS / NUMADD
local specialKeyMap = {
  -- navigation / editing
  ["ESC"]=0x1B,["ESCAPE"]=0x1B,["SPACE"]=0x20,["TAB"]=0x09,
  ["ENTER"]=0x0D,["RETURN"]=0x0D,
  ["UP"]=0x26,["DOWN"]=0x28,["LEFT"]=0x25,["RIGHT"]=0x27,
  ["HOME"]=0x24,["END"]=0x23,
  ["PGUP"]=0x21,["PAGEUP"]=0x21,["PGDN"]=0x22,["PAGEDOWN"]=0x22,
  ["INS"]=0x2D,["INSERT"]=0x2D,["DEL"]=0x2E,["DELETE"]=0x2E,
  ["BACKSPACE"]=VK_BACK,["BS"]=VK_BACK,

  -- locks / system
  ["CAPSLOCK"]=VK_CAPITAL,["CAPS"]=VK_CAPITAL,
  ["NUMLOCK"]=VK_NUMLOCK,
  ["SCROLLLOCK"]=VK_SCROLL,["SCROLL"]=VK_SCROLL,
  ["PRINTSCREEN"]=VK_SNAPSHOT,["PRTSC"]=VK_SNAPSHOT,["SNAPSHOT"]=VK_SNAPSHOT,
  ["PAUSE"]=VK_PAUSE,["BREAK"]=VK_PAUSE,
  ["LWIN"]=VK_LWIN,["RWIN"]=VK_RWIN,["WIN"]=VK_LWIN,
  ["APPS"]=VK_APPS,["MENU"]=VK_APPS,["CONTEXT"]=VK_APPS,
  ["CLEAR"]=VK_CLEAR,

  -- mouse
  ["MOUSE1"]=VK_LBUTTON,["MB1"]=VK_LBUTTON,["LMB"]=VK_LBUTTON,
  ["MOUSE2"]=VK_RBUTTON,["MB2"]=VK_RBUTTON,["RMB"]=VK_RBUTTON,
  ["MOUSE3"]=VK_MBUTTON,["MB3"]=VK_MBUTTON,["MMB"]=VK_MBUTTON,
  ["MOUSE4"]=VK_XBUTTON1,["MB4"]=VK_XBUTTON1,
  ["MOUSE5"]=VK_XBUTTON2,["MB5"]=VK_XBUTTON2,

  -- numpad ops (aliases)
  ["NUMPAD*"]=VK_MULTIPLY,["NUM*"]=VK_MULTIPLY,["NUMPADMULTIPLY"]=VK_MULTIPLY,["NUMMULTIPLY"]=VK_MULTIPLY,
  ["NUMPAD-"]=VK_SUBTRACT,["NUM-"]=VK_SUBTRACT,["NUMPADMINUS"]=VK_SUBTRACT,["NUMMINUS"]=VK_SUBTRACT,
  ["NUMPAD."]=VK_DECIMAL,["NUM."]=VK_DECIMAL,["NUMPADDOT"]=VK_DECIMAL,["NUMDOT"]=VK_DECIMAL,["NUMPADDECIMAL"]=VK_DECIMAL,
  ["NUMPAD/"]=VK_DIVIDE,["NUM/"]=VK_DIVIDE,["NUMPADDIVIDE"]=VK_DIVIDE,["NUMDIVIDE"]=VK_DIVIDE,

  -- numpad plus: IMPORTANT, support NUMPAD+ style (rewritten before splitting) + aliases
  ["NUMPADPLUS"]=VK_ADD,["NUMPADADD"]=VK_ADD,["NUMADD"]=VK_ADD,["NUMPLUS"]=VK_ADD,

  -- main keyboard punctuation
  [";"]=VK_OEM_1,["SEMICOLON"]=VK_OEM_1,
  ["'"]=VK_OEM_7,["APOSTROPHE"]=VK_OEM_7,["QUOTE"]=VK_OEM_7,
  [","]=VK_OEM_COMMA,["COMMA"]=VK_OEM_COMMA,
  ["."]=VK_OEM_PERIOD,["DOT"]=VK_OEM_PERIOD,["PERIOD"]=VK_OEM_PERIOD,
  ["/"]=VK_OEM_2,["SLASH"]=VK_OEM_2,["FORWARDSLASH"]=VK_OEM_2,
  ["\\"]=VK_OEM_5,["BACKSLASH"]=VK_OEM_5,
  ["["]=VK_OEM_4,["LBRACKET"]=VK_OEM_4,["LEFTBRACKET"]=VK_OEM_4,
  ["]"]=VK_OEM_6,["RBRACKET"]=VK_OEM_6,["RIGHTBRACKET"]=VK_OEM_6,
  ["`"]=VK_OEM_3,["GRAVE"]=VK_OEM_3,["TILDE"]=VK_OEM_3,["BACKTICK"]=VK_OEM_3,
  ["-"]=VK_OEM_MINUS,["MINUS"]=VK_OEM_MINUS,["DASH"]=VK_OEM_MINUS,["HYPHEN"]=VK_OEM_MINUS,
  ["="]=VK_OEM_PLUS,["EQUAL"]=VK_OEM_PLUS,["EQUALS"]=VK_OEM_PLUS,["PLUS"]=VK_OEM_PLUS,["OEMPLUS"]=VK_OEM_PLUS,

  -- "offline numpad" (NumLock OFF) aliases.
  -- NOTE: These map to the same VK codes as the normal navigation keys, so they can't be distinguished from the dedicated keys.
  ["NUMHOME"]=0x24,["NUMUP"]=0x26,["NUMPGUP"]=0x21,
  ["NUMLEFT"]=0x25,["NUMCLEAR"]=VK_CLEAR,["NUMRIGHT"]=0x27,
  ["NUMEND"]=0x23,["NUMDOWN"]=0x28,["NUMPGDN"]=0x22,
  ["NUMINS"]=0x2D,["NUMDEL"]=0x2E,
}

local function parseKeyToken(tok)
  tok = upperNoSpaces(tok)
  if tok == "" then return nil end

  -- single digit / letter
  if #tok == 1 and tok:match("%d") then return 0x30 + tonumber(tok) end
  if #tok == 1 and tok:match("%a") then
    local c = tok:byte()
    if c >= 0x41 and c <= 0x5A then return c end
  end

  -- single char punctuation
  if #tok == 1 and specialKeyMap[tok] then return specialKeyMap[tok] end

  -- function keys
  local fnum = tok:match("^F(%d+)$")
  if fnum then
    local n = tonumber(fnum)
    if n and n >= 1 and n <= 24 then return 0x70 + (n - 1) end
  end

  -- numpad digits
  local nd = tok:match("^NUMPAD(%d)$") or tok:match("^NUM(%d)$")
  if nd then
    local n = tonumber(nd)
    if n and n >= 0 and n <= 9 then return VK_NUMPAD0 + n end
  end

  if specialKeyMap[tok] then return specialKeyMap[tok] end
  return nil
end

local function parseKeySpec(input)
  local s = upperNoSpaces(input)
  if s == "" then return nil, nil, "Empty key" end

  -- allow "NUMPAD+" / "NUM+" even though '+' is our separator
  s = s:gsub("NUMPAD%+", "NUMPADPLUS")
  s = s:gsub("NUM%+", "NUMPADPLUS")

  -- allow trying to type the PLUS key as trailing '+', e.g. "CTRL++"
  if s:sub(-1) == "+" then
    s = s .. "PLUS"
  end

  local parts = {}
  for part in s:gmatch("[^+]+") do parts[#parts+1] = part end
  if #parts == 0 then return nil, nil, "Invalid key" end

  local mod = 0
  for i = 1, #parts - 1 do
    local m = parts[i]
    if m == "ALT" then mod = bor(mod, MOD_ALT)
    elseif m == "CTRL" or m == "CONTROL" then mod = bor(mod, MOD_CTRL)
    elseif m == "SHIFT" then mod = bor(mod, MOD_SHIFT)
    else
      return nil, nil, "Unknown modifier: " .. m
    end
  end

  local vk = parseKeyToken(parts[#parts])
  if not vk then return nil, nil, "Unknown key: " .. parts[#parts] end

  -- block binding modifier keys as the main key (generic + left/right)
  if vk == VK_ALT or vk == VK_CTRL or vk == VK_SHIFT
    or vk == VK_LSHIFT or vk == VK_RSHIFT or vk == VK_LCONTROL or vk == VK_RCONTROL
    or vk == VK_LMENU or vk == VK_RMENU
  then
    return nil, nil, "You cannot bind ALT/CTRL/SHIFT as the main key"
  end

  return vk, mod, nil
end

local function parseItem(str)
  if type(str) ~= "string" then return nil end
  local a, b, c = str:match("^(%d+)|(%d+)|(.+)$")
  if not a or not b or not c then return nil end
  return tonumber(a), tonumber(b), c
end

local function makeItem(vk, mod, cmd)
  return tostring(vk) .. "|" .. tostring(mod) .. "|" .. cmd
end

local function iterItems()
  local out = {}
  local count = tonumber(cfg.meta.count) or 0
  for i = 1, count do
    local raw = cfg.items[tostring(i)]
    local vk, mod, cmd = parseItem(raw)
    if vk and cmd then out[#out+1] = { idx=i, vk=vk, mod=mod or 0, cmd=cmd } end
  end
  return out
end

local function findByKey(vk, mod)
  for _, it in ipairs(iterItems()) do
    if it.vk == vk and (it.mod or 0) == (mod or 0) then return it end
  end
  return nil
end

local function writeTxt()
  local f = io.open(BIND_TXT_PATH, "w")
  if not f then return end
  for _, it in ipairs(iterItems()) do
    f:write(makeItem(it.vk, it.mod or 0, it.cmd), "\n")
  end
  f:close()
end

local function loadTxtIfExists()
  local f = io.open(BIND_TXT_PATH, "r")
  if not f then return false end

  local lines = {}
  for line in f:lines() do
    line = trim(line)
    if line ~= "" then lines[#lines+1] = line end
  end
  f:close()
  if #lines == 0 then return false end

  cfg.items = {}
  cfg.meta.count = 0
  for i, line in ipairs(lines) do
    local vk, mod, cmd = parseItem(line)
    if vk and cmd then
      cfg.meta.count = i
      cfg.items[tostring(i)] = makeItem(vk, mod or 0, cmd)
    end
  end
  saveIni()
  return true
end

local function addOrUpdateBind(vk, mod, cmd)
  local existing = findByKey(vk, mod)
  if existing then
    cfg.items[tostring(existing.idx)] = makeItem(vk, mod, cmd)
    saveIni(); writeTxt()
    return "Updated", existing.idx
  end
  local count = (tonumber(cfg.meta.count) or 0) + 1
  cfg.meta.count = count
  cfg.items[tostring(count)] = makeItem(vk, mod, cmd)
  saveIni(); writeTxt()
  return "Added", count
end

local function removeBindByKey(vk, mod)
  local it = findByKey(vk, mod)
  if not it then return false end
  local count = tonumber(cfg.meta.count) or 0
  for i = it.idx, count - 1 do
    cfg.items[tostring(i)] = cfg.items[tostring(i + 1)]
  end
  cfg.items[tostring(count)] = nil
  cfg.meta.count = math.max(0, count - 1)
  saveIni(); writeTxt()
  return true
end

local function clearAll()
  cfg.items = {}
  cfg.meta.count = 0
  saveIni(); writeTxt()
end

-- =========================
-- UI
-- =========================
local DLG_VIEW = 18100
local DLG_KEYS = 18101
local DLG_STYLE_TABLIST_HEADERS = 5

local function showKeybindsDialog()
  local items = iterItems()
  table.sort(items, function(a, b) return keyDisplay(a.vk, a.mod) < keyDisplay(b.vk, b.mod) end)

  local text = "{00FF90}Key\tCommand{FFFFFF}\n"
  if #items == 0 then
    text = text .. "None\tUse /keybind [key] [bind]\n"
  else
    for _, it in ipairs(items) do
      text = text .. string.format("%s\t%s\n", keyDisplay(it.vk, it.mod), it.cmd)
    end
  end

  text = text .. "\n{AAAAAA}Tip\tType /keys to see valid key names{FFFFFF}\n"
  text = text .. "{AAAAAA}Tip\t/keybind [key] [bind] (example: /keybind ALT+S /use adrenaline syringe){FFFFFF}\n"
  text = text .. "{AAAAAA}Tip\t/keyunbind [key] (example: /keyunbind ALT+S){FFFFFF}\n"

  sampShowDialog(DLG_VIEW, "Current Keybinds", text, "Close", "", DLG_STYLE_TABLIST_HEADERS)
end

local function showKeysGuideDialog()
  -- Keep this compact so it fits in SA-MP dialog limits.
  local text = "{00FF90}What to type\tMeaning / Notes{FFFFFF}\n"

  text = text .. "ALT+S\tHold ALT and press S\n"
  text = text .. "CTRL+X\tHold CTRL and press X\n"
  text = text .. "SHIFT+F2\tHold SHIFT and press F2\n"
  text = text .. "ALT+CTRL+Z\tHold ALT+CTRL and press Z\n"

  text = text .. "\n{00FF90}Letters / Numbers\t{FFFFFF}\n"
  text = text .. "A..Z\tLetters\n"
  text = text .. "0..9\tTop-row numbers\n"

  text = text .. "\n{00FF90}Function keys\t{FFFFFF}\n"
  text = text .. "F1..F24\tFunction keys\n"

  text = text .. "\n{00FF90}Mouse buttons\t{FFFFFF}\n"
  text = text .. "MOUSE1..MOUSE5\tLeft/Right/Middle/Extra1/Extra2 (also: LMB/RMB/MMB/MB1..MB5)\n"

  text = text .. "\n{00FF90}Numpad (NumLock ON)\t{FFFFFF}\n"
  text = text .. "NUMPAD0..NUMPAD9\tDigits (also: NUM0..NUM9)\n"
  text = text .. "NUMPAD*\tMultiply (also: NUM*)\n"
  text = text .. "NUMPAD-\tMinus (also: NUM-)\n"
  text = text .. "NUMPAD/\tDivide (also: NUM/)\n"
  text = text .. "NUMPAD.\tDecimal (also: NUM.)\n"
  text = text .. "NUMPAD+\tPlus: type NUMPAD+ or NUMPADPLUS / NUMADD\n"

  text = text .. "\n{00FF90}Offline numpad (NumLock OFF)\t{FFFFFF}\n"
  text = text .. "NUMHOME/NUMUP/NUMPGUP\tSame VK as HOME/UP/PGUP\n"
  text = text .. "NUMLEFT/NUMCLEAR/NUMRIGHT\tSame VK as LEFT/CLEAR/RIGHT\n"
  text = text .. "NUMEND/NUMDOWN/NUMPGDN\tSame VK as END/DOWN/PGDN\n"
  text = text .. "NUMINS/NUMDEL\tSame VK as INS/DEL\n"

  text = text .. "\n{00FF90}Navigation / Editing\t{FFFFFF}\n"
  text = text .. "ESC, TAB, ENTER, SPACE, BACKSPACE\tCommon keys\n"
  text = text .. "UP/DOWN/LEFT/RIGHT\tArrow keys\n"
  text = text .. "HOME/END, PGUP/PGDN\tNavigation\n"
  text = text .. "INS/DEL\tInsert / Delete\n"

  text = text .. "\n{00FF90}System / Locks\t{FFFFFF}\n"
  text = text .. "CAPSLOCK, NUMLOCK, SCROLLLOCK\tLock keys\n"
  text = text .. "PRTSC, PAUSE\tPrint Screen / Pause\n"
  text = text .. "LWIN, RWIN, APPS\tWindows keys / context menu\n"

  text = text .. "\n{00FF90}Punctuation / special characters\t{FFFFFF}\n"
  text = text .. "[ ] \\ / ; ' , . - `\tYou can type the character itself\n"
  text = text .. "EQUALS or PLUS\tBind the =/+ key (don't type '+')\n"
  text = text .. "Aliases\tLBRACKET/RBRACKET, BACKSLASH, SLASH, SEMICOLON, QUOTE, COMMA, PERIOD, MINUS, GRAVE\n"

  sampShowDialog(DLG_KEYS, "Keys Guide", text, "Close", "", DLG_STYLE_TABLIST_HEADERS)
end

-- =========================
-- Main loop
-- =========================
-- Send bind text:
--  - If it starts with '/', try to process as a local command first (other Lua scripts),
--    then fall back to sending to server.
--  - Otherwise, send as normal chat text.
local function sendBindText(text)
  text = tostring(text or "")
  if text == "" then return end

  if text:sub(1, 1) == "/" then
    if type(sampProcessChatInput) == "function" then
      local ok, handled = pcall(sampProcessChatInput, text)
      -- If the function exists but did not handle it (or errored), send to server.
      if not ok or handled == false then
        sampSendChat(text)
      end
    else
      -- Older builds without sampfuncs
      sampSendChat(text)
    end
  else
    sampSendChat(text)
  end
end

local function currentHeldMods()
  local m = 0
  if isKeyDown(VK_ALT) then m = bor(m, MOD_ALT) end
  if isKeyDown(VK_CTRL) then m = bor(m, MOD_CTRL) end
  if isKeyDown(VK_SHIFT) then m = bor(m, MOD_SHIFT) end
  return m
end

local function popcount(x)
  x = tonumber(x) or 0
  local c = 0
  while x ~= 0 do
    c = c + 1
    x = band(x, x - 1)
  end
  return c
end

local function processBinds()
  if isInputBlocked() then return end

  local items = iterItems()
  if #items == 0 then return end

  local held = currentHeldMods()

  -- If multiple binds share the same base key (e.g. MOUSE5 and ALT+MOUSE5),
  -- pick the most specific one that matches the currently-held modifiers.
  local unique = {}
  for _, it in ipairs(items) do
    unique[it.vk] = true
  end

  for vk, _ in pairs(unique) do
    if wasKeyPressed(vk) then
      local best, bestScore = nil, -1
      for _, it in ipairs(items) do
        if it.vk == vk then
          local req = it.mod or 0
          if band(held, req) == req then
            local score = popcount(req)
            if score > bestScore then
              best, bestScore = it, score
            end
          end
        end
      end

      if best then
        sendBindText(best.cmd)
      end
    end
  end
end

function main()
  repeat wait(0) until isSampAvailable()

  loadTxtIfExists()

  sampRegisterChatCommand("keybinds", showKeybindsDialog)
  sampRegisterChatCommand("keys", showKeysGuideDialog)

  sampRegisterChatCommand("keybind", function(params)
    params = trim(params)
    if params == "" then
      sampAddChatMessage("{FFCC00}Usage:{FFFFFF} /keybind [key] [bind]", -1)
      return
    end

    local keyPart, cmdPart = params:match("^(%S+)%s+(.+)$")
    if not keyPart or not cmdPart then
      sampAddChatMessage("{FF6666}[Keybinds]{FFFFFF} Invalid format. Usage: /keybind [key] [bind]", -1)
      return
    end

    local vk, mod, err = parseKeySpec(keyPart)
    if err then
      sampAddChatMessage("{FF6666}[Keybinds]{FFFFFF} " .. err .. "  |  Type /keys", -1)
      return
    end

    cmdPart = trim(cmdPart)
    -- Do NOT force a leading '/'. Allow normal chat text binds too.

    local action, idx = addOrUpdateBind(vk, mod, cmdPart)
    sampAddChatMessage(string.format("{00FF90}[Keybinds]{FFFFFF} %s: %s -> %s (slot %d).",
      action, keyDisplay(vk, mod), cmdPart, idx
    ), -1)
  end)

  sampRegisterChatCommand("keyunbind", function(params)
    params = trim(params)
    if params == "" then
      sampAddChatMessage("{FFCC00}Usage:{FFFFFF} /keyunbind [key]", -1)
      return
    end

    local vk, mod, err = parseKeySpec(params)
    if err then
      sampAddChatMessage("{FF6666}[Keybinds]{FFFFFF} " .. err .. "  |  Type /keys", -1)
      return
    end

    if removeBindByKey(vk, mod) then
      sampAddChatMessage(string.format("{00FF90}[Keybinds]{FFFFFF} Removed bind: %s.", keyDisplay(vk, mod)), -1)
    else
      sampAddChatMessage(string.format("{FF6666}[Keybinds]{FFFFFF} No bind found for: %s.", keyDisplay(vk, mod)), -1)
    end
  end)

  sampRegisterChatCommand("keyclear", function()
    clearAll()
    sampAddChatMessage("{00FF90}[Keybinds]{FFFFFF} Cleared all keybinds.", -1)
  end)

  sampRegisterChatCommand("keyblock", function()
    cfg.settings.block_when_input = (tonumber(cfg.settings.block_when_input) == 1) and 0 or 1
    saveIni()
    sampAddChatMessage(string.format("{00FF90}[Keybinds]{FFFFFF} Block while typing/dialog: %s.",
      (tonumber(cfg.settings.block_when_input) == 1) and "ON" or "OFF"
    ), -1)
  end)

  while true do
    wait(0)
    processBinds()
  end
end