script_name("AutoFPSLimiter")
script_author("ncta")

-- Make sure MoonLoader core is loaded
require "lib.moonloader"

-------------------------------------------------
-- Simple persistent config (no external libs)
-- Saves next to this script: AutoFPSLimiter.cfg
-------------------------------------------------
local function getConfigPath()
    if thisScript and type(thisScript) == "function" then
        local p = thisScript().path or ""
        local dir = p:match("^(.*\\)") or ""
        if dir ~= "" then
            return dir .. "AutoFPSLimiter.cfg"
        end
    end
    -- Fallback: current working dir
    return "AutoFPSLimiter.cfg"
end

local CFG_PATH = getConfigPath()

local function loadCfg()
    local t = {}
    local f = io.open(CFG_PATH, "r")
    if not f then return t end
    for line in f:lines() do
        local k, v = line:match("^%s*([%w_]+)%s*=%s*(.-)%s*$")
        if k and v then
            local num = tonumber(v)
            t[k] = num ~= nil and num or v
        end
    end
    f:close()
    return t
end

local function saveCfg(tbl)
    local f = io.open(CFG_PATH, "w+")
    if not f then return false end
    for k, v in pairs(tbl) do
        f:write(("%s=%s\n"):format(k, tostring(v)))
    end
    f:close()
    return true
end

local function clampFps(v)
    v = tonumber(v)
    if not v then return nil end
    v = math.floor(v + 0.5)
    if v < 0 then v = 0 end
    if v > 999 then v = 999 end
    return v
end

-------------------------------------------------
-- Settings
-------------------------------------------------
local font_name = "Verdana"
local font_size = 12
local font_flags = 12

local drive_fps = 90
local boat_fps = 28
local bike_fps = 100
local swim_fps = 20
local parachute_fps = 100
local default_fps = 0

-- Load saved default_fps (if any)
do
    local cfg = loadCfg()
    if tonumber(cfg.default_fps) then
        default_fps = clampFps(cfg.default_fps) or default_fps
    end
end

local lastState = ""

local boat_ids = {
    [430] = true, [446] = true, [452] = true, [453] = true,
    [454] = true, [472] = true, [473] = true, [484] = true,
    [493] = true, [595] = true
}

local bike_ids = {
    [509] = true, [481] = true, [510] = true,
    [462] = true, [521] = true, [463] = true, [522] = true,
    [461] = true, [448] = true, [468] = true, [586] = true,
    [471] = true, [523] = true, [581] = true
}

local hasFFI, ffi = pcall(require, "ffi")
local SetFrameRate = nil

if hasFFI then
    ffi.cdef[[
        void* GetModuleHandleA(const char* lpModuleName);
        void* GetProcAddress(void* hModule, const char* lpProcName);
    ]]
    local hModule = ffi.C.GetModuleHandleA("PreciseFramerateLimiter.asi")
    if hModule ~= nil then
        local procAddr = ffi.C.GetProcAddress(hModule, "PreciseFramerateLimiter_SetFrameRate")
        if procAddr ~= nil then
            SetFrameRate = ffi.cast("void (__cdecl*)(int)", procAddr)
        end
    end
end

-- Script state
local scriptEnabled = true
local labelEnabled = false

-- Font
local Font = nil

local function isCharDriver(ped, veh)
    local driver = getDriverOfCar(veh)
    return doesCharExist(driver) and driver == ped
end

local function applyFps(state, val)
    if lastState ~= state then
        if SetFrameRate then
            SetFrameRate(val)
        else
            sampProcessChatInput("/fpslimit " .. val)
        end
        lastState = state
    end
end

function main()
    while not isSampAvailable() do wait(500) end

    Font = renderCreateFont(font_name, font_size, font_flags)

    -------------------------------------------------
    -- Commands
    -------------------------------------------------
    sampRegisterChatCommand("fpstoggle", function()
        scriptEnabled = not scriptEnabled
        sampAddChatMessage("[AutoFPSLimiter] " .. (scriptEnabled and "Enabled" or "Disabled"), -1)
        if not scriptEnabled then
            lastState = ""
            applyFps("default", default_fps)
        end
    end)

    sampRegisterChatCommand("fpslabel", function()
        labelEnabled = not labelEnabled
        sampAddChatMessage("[AutoFPSLimiter] Label " .. (labelEnabled and "ON" or "OFF"), -1)
    end)

    -- /fpsdefault 0..999  -> changes in real time and saves to AutoFPSLimiter.cfg (0 = unlimited)
    sampRegisterChatCommand("fpsdefault", function(arg)
        local v = clampFps(arg)
        if not v then
            sampAddChatMessage("[AutoFPSLimiter] Usage: /fpsdefault <0-999> (0 = unlimited)", -1)
            return
        end

        default_fps = v

        saveCfg({ default_fps = default_fps })

        lastState = ""
        applyFps("default", default_fps)

        sampAddChatMessage("[AutoFPSLimiter] default_fps set to " .. default_fps .. " and saved.", -1)
    end)

    sampRegisterChatCommand("fpsstatus", function()
        sampAddChatMessage(string.format(
            "[AutoFPSLimiter] default=%d drive=%d bike=%d boat=%d swim=%d parachute=%d | label=%s | script=%s",
            default_fps, drive_fps, bike_fps, boat_fps, swim_fps, parachute_fps,
            labelEnabled and "ON" or "OFF",
            scriptEnabled and "ON" or "OFF"
        ), -1)
    end)

    local lastFpsCheck = os.clock()

    while true do
        wait(0)

        if scriptEnabled and (os.clock() - lastFpsCheck >= 0.5) then
            lastFpsCheck = os.clock()
            local ped = PLAYER_PED

            if isCharInWater(ped) then
                applyFps("swim", swim_fps)

            elseif getCurrentCharWeapon(ped) == 46 then
                applyFps("parachute", parachute_fps)

            elseif isCharInAnyCar(ped) then
                local veh = storeCarCharIsInNoSave(ped)
                if isCharDriver(ped, veh) then
                    local model = getCarModel(veh)
                    if boat_ids[model] then
                        applyFps("boat", boat_fps)
                    elseif bike_ids[model] then
                        applyFps("bike", bike_fps)
                    else
                        applyFps("drive", drive_fps)
                    end
                else
                    applyFps("passenger", default_fps)
                end

            else
                applyFps("default", default_fps)
            end
        end

        if labelEnabled then
            local sw, sh = getScreenResolution()
            local status = scriptEnabled and "AutoFPSLimiter: ON" or "AutoFPSLimiter: OFF"
            local tw = renderGetFontDrawTextLength(Font, status)
            renderFontDrawText(Font, status, sw - tw - 10, sh - font_size - 10, 0xFFFFFFFF)
        end
    end
end