script_name("AllianceList")
script_author("ncta")
script_version("1.7")

local sampev = require "lib.samp.events"

-- config
local OUR_GROUP = "MCF"

-- colors (match tlist vibe)
local WHITE = "{FFFFFF}"
local GREY  = "{808080}"
local LIGHT_GREY = "{C0C0C0}"
local GREEN = "{4CAF50}"
local BLUE  = "{4C8DFF}"
local YELLOW = "{D4A300}"
local PINK  = "{FFFFFF}" -- (was pink, now white)
local ORANGE = "{FFAA00}"
local FOOTER = "{6C7887}" -- footer / bottom line

-- state
local isProcessing = false
local allianceDialogText = ""
local crimsDialogText = ""
local processingStep = 0 -- 0: idle, 1: getting alliance, 2: getting crims

-- helpers
local function strip_colors(s)
    return (s or ""):gsub("{%x%x%x%x%x%x}", "")
end

local function trim(s)
    s = tostring(s or "")
    s = s:gsub("\r", "")
    s = s:gsub("^%s+", ""):gsub("%s+$", "")
    return s
end

local function eqTag(a, b)
    return tostring(a or ""):upper() == tostring(b or ""):upper()
end

local function padRight(s, width)
    s = tostring(s or "")
    local len = #s
    if len >= width then return s end
    return s .. string.rep(" ", width - len)
end

local function fmtAllianceTitle(name, tag)
    name = trim(strip_colors(name))
    tag  = trim(strip_colors(tag))
    if name == "" and tag ~= "" then return tag end
    if tag ~= "" then return string.format("%s (%s)", name, tag) end
    return name
end

local function allianceHasOurGroup(members)
    for _, m in ipairs(members or {}) do
        if eqTag(m.tag, OUR_GROUP) then return true end
    end
    return false
end

local function sortMembers(members)
    table.sort(members, function(a, b)
        local A = eqTag(a.tag, OUR_GROUP)
        local B = eqTag(b.tag, OUR_GROUP)
        if A ~= B then return A end

        local ao = a.isOwner and 1 or 0
        local bo = b.isOwner and 1 or 0
        if ao ~= bo then return ao > bo end

        local aOn = tonumber(a.online or 0) or 0
        local bOn = tonumber(b.online or 0) or 0
        if aOn ~= bOn then return aOn > bOn end

        return tostring(a.tag or ""):lower() < tostring(b.tag or ""):lower()
    end)
end

local function formatAllianceBlock(allianceName, allianceTag, members, padWidth)
    -- compute total online (optional use)
    local totalOnline = 0
    for _, m in ipairs(members or {}) do
        totalOnline = totalOnline + (tonumber(m.online or 0) or 0)
    end

    local lines = {}
    local title = fmtAllianceTitle(allianceName, allianceTag)

    -- Header: blue only for OUR alliance; other alliances stay default color
    local hasOur = allianceHasOurGroup(members)
    local headerTitleColor = hasOur and BLUE or WHITE

    local totalColor = (totalOnline > 0) and GREEN or GREY
    -- "online" should match the total number color (Solo Groups header stays white)
    table.insert(lines, string.format("%s-> %s%s%s %s- %s%d%s %sonline%s", WHITE, headerTitleColor, title, WHITE, LIGHT_GREY, totalColor, totalOnline, WHITE, totalColor, WHITE))
    -- One-line member list: "-> MCF (10), VCS (12)"
    -- keep our group first if present, then sort the rest
    local ordered = {}

    -- keep our group first if present, then sort the rest
    if hasOur then
        for _, m in ipairs(members or {}) do
            if eqTag(trim(m.tag), OUR_GROUP) then
                table.insert(ordered, m)
            end
        end
        local rest = {}
        for _, m in ipairs(members or {}) do
            if not eqTag(trim(m.tag), OUR_GROUP) then
                table.insert(rest, m)
            end
        end
        sortMembers(rest)
        for _, m in ipairs(rest) do table.insert(ordered, m) end
    else
        ordered = members or {}
        sortMembers(ordered)
    end

    local parts = {}
    for _, m in ipairs(ordered or {}) do
        local tag = trim(m.tag)
        if tag ~= "" then
            local on = tonumber(m.online or 0) or 0

            local tagColor = PINK
local displayColor = (on > 0) and tagColor or WHITE

            -- tag text stays in tagColor; make the "( ... )" light grey, but keep the number white
            local parenColor = WHITE
            local numColor   = (on > 0) and YELLOW or GREY

            table.insert(parts, string.format("%s%s%s (%s%d%s)%s", displayColor, tag, parenColor, numColor, on, parenColor, WHITE))
        end
    end

    if #parts > 0 then
        -- arrow is for the alliance, not for the groups
        table.insert(lines, string.format("%s   %s", WHITE, table.concat(parts, string.format("%s, %s", GREY, WHITE))))
    end


    return table.concat(lines, "\n")
end

function main()
    while not isSampAvailable() do wait(500) end

    sampRegisterChatCommand("alist", function()
        if isProcessing then return end
        isProcessing = true
        processingStep = 1
        allianceDialogText = ""
        crimsDialogText = ""
        sampSendChat("/alliance list")
    end)

    wait(-1)
end

function sampev.onShowDialog(dialogId, style, title, button1, button2, text)
    local t = strip_colors(title or "")

    if isProcessing and t == "Alliances" and processingStep == 1 then
        allianceDialogText = text or ""
        processingStep = 2

        lua_thread.create(function()
            wait(450)
            sampSendChat("/crims")
        end)

        return false
    elseif isProcessing and t == "Criminal Groups" and processingStep == 2 then
        crimsDialogText = text or ""

        lua_thread.create(function()
            wait(50)
            parseAndDisplayAlliancesWithOnline()
            isProcessing = false
            processingStep = 0
        end)

        return false
    end
end

function parseAndDisplayAlliancesWithOnline()
    if allianceDialogText == "" or crimsDialogText == "" then
        sampAddChatMessage("{FF0000}[alist]{FFFFFF} Missing dialog data.", -1)
        return
    end

    -- build online counts from /crims
    local onlineCounts = {}
    local crimsOrder = {}
    local totalOnlineAll = 0
    for line in (crimsDialogText or ""):gmatch("[^\r\n]+") do
        local clean = strip_colors(line)
        -- TAG  <online>  <members> ...
        local tag, onlineCount = clean:match("([A-Z0-9_]+)%s+(%d+)%s+%d+")
        if tag and onlineCount then
            local on = tonumber(onlineCount) or 0
            if onlineCounts[tag] == nil then table.insert(crimsOrder, tag) end
            onlineCounts[tag] = on
            totalOnlineAll = totalOnlineAll + on
        end
    end

    -- parse /alliance list
    local allianceLines = {}
    for line in (allianceDialogText or ""):gmatch("[^\r\n]+") do
        table.insert(allianceLines, line)
    end

    local currentAlliance = nil
    local currentAllianceTag = nil
    local currentMembers = {}
    local alliances = {}
    local groupsInAlliances = {}

    local function pushCurrent()
        if currentAlliance and #currentMembers > 0 then
            table.insert(alliances, { name = currentAlliance, tag = currentAllianceTag, members = currentMembers })
        end
    end

    for _, line in ipairs(allianceLines) do
        local cleanLine = strip_colors(line)

        -- member lines are indented
        if cleanLine:match("^%s+") then
            local _, memberTag = cleanLine:match("^%s*(.-)%s*%[(.-)%]")
            if memberTag and memberTag ~= "" then
                local isOwner = cleanLine:find("%[Owner%]", 1, true) ~= nil
                local on = onlineCounts[memberTag] or 0
                groupsInAlliances[memberTag] = true
                table.insert(currentMembers, { tag = memberTag, online = on, isOwner = isOwner })
            end
        else
            -- new alliance header
            pushCurrent()

            local allianceName, allianceTag = cleanLine:match("^(.-)%s*%[(.-)%]")
            if allianceName and allianceTag then
                currentAlliance = trim(allianceName)
                currentAllianceTag = trim(allianceTag)
                currentMembers = {}
            end
        end
    end
    pushCurrent()

    -- sort: OUR alliance first, then by total online desc
    table.sort(alliances, function(a, b)
        local aHas = allianceHasOurGroup(a.members)
        local bHas = allianceHasOurGroup(b.members)
        if aHas ~= bHas then return aHas end

        local aTotal, bTotal = 0, 0
        for _, m in ipairs(a.members or {}) do aTotal = aTotal + (tonumber(m.online or 0) or 0) end
        for _, m in ipairs(b.members or {}) do bTotal = bTotal + (tonumber(m.online or 0) or 0) end
        if aTotal ~= bTotal then return aTotal > bTotal end

        return tostring(a.name or ""):lower() < tostring(b.name or ""):lower()
    end)

    -- solo groups (present in /crims but not inside any alliance)
    local soloGroups = {}
    local soloTotalOnline = 0
    local _idx = 0
    for _, tag in ipairs(crimsOrder) do
        _idx = _idx + 1
        local on = onlineCounts[tag] or 0
        if not groupsInAlliances[tag] then
            table.insert(soloGroups, { tag = tag, online = on, idx = _idx })
            soloTotalOnline = soloTotalOnline + (tonumber(on or 0) or 0)
        end
    end

    -- sort: OUR group first, then by online desc (stable by original /crims order)
    table.sort(soloGroups, function(a, b)
        local aIs = eqTag(a.tag, OUR_GROUP)
        local bIs = eqTag(b.tag, OUR_GROUP)
        if aIs ~= bIs then return aIs end

        local aOn = tonumber(a.online or 0) or 0
        local bOn = tonumber(b.online or 0) or 0
        if aOn ~= bOn then return aOn > bOn end

        return (tonumber(a.idx or 0) or 0) < (tonumber(b.idx or 0) or 0)
    end)

    -- padding width for tags (for nicer alignment)
    local maxTagLen = 3
    for _, al in ipairs(alliances) do
        for _, m in ipairs(al.members or {}) do
            maxTagLen = math.max(maxTagLen, #(tostring(m.tag or "")))
        end
    end
    for _, g in ipairs(soloGroups) do
        maxTagLen = math.max(maxTagLen, #(tostring(g.tag or "")))
    end
    maxTagLen = math.min(maxTagLen, 10) -- keep it sane

    -- build dialog
    local out = {}
    table.insert(out, string.format("%s%s%s", GREY, os.date("%H:%M:%S"), WHITE))
    table.insert(out, "")

    for _, al in ipairs(alliances) do
        table.insert(out, formatAllianceBlock(al.name, al.tag, al.members, maxTagLen))
        table.insert(out, "")
    end

    if #soloGroups > 0 then
                local soloColor = (soloTotalOnline > 0) and WHITE or GREY
                table.insert(out, string.format("%sSolo Groups %s- %s%d%s %sonline", WHITE, LIGHT_GREY, soloColor, soloTotalOnline, WHITE, WHITE))



        for _, g in ipairs(soloGroups) do
            local tag = trim(g.tag)
            local on  = tonumber(g.online or 0) or 0
            local tagColor = PINK
            if eqTag(tag, OUR_GROUP) then tagColor = BLUE end

            local displayColor
            if eqTag(tag, OUR_GROUP) then
                displayColor = tagColor
            else
                displayColor = (on > 0) and tagColor or WHITE
            end

            -- same format as allied groups: "TAG (7)"
            local parenColor = WHITE
            local numColor   = (on > 0) and YELLOW or GREY

            -- arrow is for the Solo Groups header, not for each group
            table.insert(out, string.format("%s   %s%s%s (%s%d%s)%s", WHITE, displayColor, tag, parenColor, numColor, on, parenColor, WHITE))
        end

        table.insert(out, "")
    end

    -- footer
    table.insert(out, string.format("%sTotal: %d alliances | %d solo groups | %d online", FOOTER, #alliances, #soloGroups, totalOnlineAll))

    sampShowDialog(1337, "{FFFFFF}Alliance List", table.concat(out, "\n"), "Close", "", 0)
end